using System;
using System.Collections.Generic;
using System.Text;

namespace HL7 {
    public class HL7Message {
        #region Private Instance Variables
        //The int is the segment's location within the message
        private SortedDictionary<int, Segment> _HL7Segments = new SortedDictionary<int, Segment>();
        #endregion
        #region Public Instance Methods
        public SortedDictionary<int, T> RetrieveSegmentSequence<T>()where T:Segment {
            SortedDictionary<int, T> dictionary = new SortedDictionary<int, T>();
            foreach (KeyValuePair<int, Segment> pair in this.HL7Segments) {
                if (pair.Value.GetType() == typeof(T)) {
                    dictionary.Add(pair.Key, (T)pair.Value);
                }
            }
            return dictionary;
        }
        #endregion
        #region Public Instance Methods
        public override string ToString() {
            return KnownSegmentsHL7Message;
        }
        #endregion
        #region Public Instance Properties
        public SortedDictionary<int, MessageHeaderSegment> MessageHeaderSequence {
            get {
                return RetrieveSegmentSequence<MessageHeaderSegment>();
            }
        }
        public SortedDictionary<int, PatientIdentificationSegment> PatientIdentificationSequence {
            get {
                return RetrieveSegmentSequence<PatientIdentificationSegment>();
            }
        }
        public SortedDictionary<int, PatientVisitSegment> PatientVisitSequence {
            get {
                return RetrieveSegmentSequence<PatientVisitSegment>();
            }
        }
        public SortedDictionary<int, InsuranceSegment> InsuranceSequence {
            get {
                return RetrieveSegmentSequence<InsuranceSegment>();
            }
        }
        public SortedDictionary<int, GuarantorSegment> GuarantorSequence {
            get {
                return RetrieveSegmentSequence<GuarantorSegment>();
            }
        }
        public SortedDictionary<int, DiagnosisSegment> DiagnosisSequence {
            get {
                return RetrieveSegmentSequence<DiagnosisSegment>();
            }
        }
        public SortedDictionary<int, PatientGeneralClinicalInformationSegment> PatientGeneralClinicalInformationSequence {
            get {
                return RetrieveSegmentSequence<PatientGeneralClinicalInformationSegment>();
            }
        }
        public SortedDictionary<int, CommonOrderSegment> CommonOrderSequence {
            get {
                return RetrieveSegmentSequence<CommonOrderSegment>();
            }
        }
        public SortedDictionary<int, ObservationOrderSegment> ObservationOrderSequence {
            get {
                return RetrieveSegmentSequence<ObservationOrderSegment>();
            }
        }
        public SortedDictionary<int, BloodLeadSegment> BloodLeadSequence {
            get {
                return RetrieveSegmentSequence<BloodLeadSegment>();
            }
        }
        public SortedDictionary<int, BethesdaCytologySegment> BethesdaCytologySequence {
            get {
                return RetrieveSegmentSequence<BethesdaCytologySegment>();
            }
        }
        public SortedDictionary<int, SuperAlphaFetaProteinSegment> SuperAlphaFetaProteinSequence {
            get {
                return RetrieveSegmentSequence<SuperAlphaFetaProteinSegment>();
            }
        }
        public SortedDictionary<int, NotesAndCommentsSegment> NotesAndCommentsSequence {
            get {
                return RetrieveSegmentSequence<NotesAndCommentsSegment>();
            }
        }
        public SortedDictionary<int, AddendumSegment> AddendumSequence {
            get {
                return RetrieveSegmentSequence<AddendumSegment>();
            }
        }
        public SortedDictionary<int, ObservationResultSegment> ObservationResultSequence {
            get {
                return RetrieveSegmentSequence<ObservationResultSegment>();
            }
        }
        public SortedDictionary<int, PlaceOfServiceFacilitySegment> PlaceOfServiceFacilitySequence {
            get {
                return RetrieveSegmentSequence<PlaceOfServiceFacilitySegment>();
            }
        }
        public SortedDictionary<int, EmbeddedImageSegment> EmbeddedImageSequence {
            get {
                return RetrieveSegmentSequence<EmbeddedImageSegment>();
            }
        }
        public SortedDictionary<int, Segment> HL7Segments {
            get {
                return _HL7Segments;
            }
            private set {
                _HL7Segments = value;
            }
        }
        public string KnownSegmentsHL7Message {
            get {
                StringBuilder builder = new StringBuilder();
                foreach (KeyValuePair<int, Segment> pair in this.HL7Segments) {
                    builder.Append((string)pair.Value.GetType().GetProperty("KnownFieldsHL7String").GetValue(pair.Value, null));
                }
                return builder.ToString();
            }
        }
        public string AllSegmentsHL7Message {
            get {
                StringBuilder builder = new StringBuilder();
                foreach (KeyValuePair<int, Segment> pair in this.HL7Segments) {
                    builder.Append((string)pair.Value.GetType().GetProperty("AllFieldsHL7String").GetValue(pair.Value, null));
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructor
        public HL7Message(string message) {
            if (message.Contains("\r\n"))
                message = message.Replace("\r\n", "\r");
            if (message.Contains("\n"))
                message = message.Replace("\n", "\r");
            if (message.Contains(new MessageHeaderSegment().SegmentType) == false) {
                throw new ArgumentException("No Message Header Segment (MSH) in HL7 message.");
            }
            List<string> segments = new List<string>(message.Split(new string[] { "\r" }, StringSplitOptions.RemoveEmptyEntries));
            //So we have the field/component delimiters for the other segments
            MessageHeaderSegment messageHeader = new MessageHeaderSegment();
            Int32 x = 1;
            foreach (String segment in segments) {
                if (segment.Length < 3) {                 
                    System.Diagnostics.Debug.WriteLine("Segment is too small (less than 3 characters): " + segment);
                    continue;
                }
                switch (segment.Substring(0, 3).ToUpper()) {
                    case "MSH":
                        messageHeader = new MessageHeaderSegment(segment);
                        this.HL7Segments.Add(x, messageHeader);
                        break;
                    case "PID":
                        this.HL7Segments.Add(x, new PatientIdentificationSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "ORC":
                        this.HL7Segments.Add(x, new CommonOrderSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "OBR":
                        this.HL7Segments.Add(x, new ObservationOrderSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "OBX":
                        this.HL7Segments.Add(x, new ObservationResultSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "NTE":
                        this.HL7Segments.Add(x, new NotesAndCommentsSegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "ZPS":
                        this.HL7Segments.Add(x, new PlaceOfServiceFacilitySegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "ADD":
                        this.HL7Segments.Add(x, new AddendumSegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "PV1":
                        this.HL7Segments.Add(x, new PatientVisitSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "IN1":
                        this.HL7Segments.Add(x, new InsuranceSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "GT1":
                        this.HL7Segments.Add(x, new GuarantorSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "DG1":
                        this.HL7Segments.Add(x, new DiagnosisSegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "ZCI":
                        this.HL7Segments.Add(x, new PatientGeneralClinicalInformationSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "ZSA":
                        this.HL7Segments.Add(x, new SuperAlphaFetaProteinSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "ZCY":
                        this.HL7Segments.Add(x, new BethesdaCytologySegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "ZBL":
                        this.HL7Segments.Add(x, new BloodLeadSegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "ZEF":
                        this.HL7Segments.Add(x, new EmbeddedImageSegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    default:
                        throw new ArgumentException("Unknown HL7 string passed to HL7TypeFromString");
                }
                x++;
            }
        }
        public HL7Message() { }
        #endregion
    }
}