using System;
using System.Collections.Generic;
using System.Text;

namespace HL7 {
    public class MessageHeaderSegment : Segment {
        #region Private Instance Variables
        private Field _FieldDelimiter = new Field();
        private Field _ComponentDelimiter = new Field();
        private Field _RepeatDelimiter = new Field();
        private Field _EscapeCharacter = new Field();
        private Field _SubComponentDelimiter = new Field();
        private Field _SendingApplication = new Field();
        private Field _SendingFacility = new Field();
        private Field _ReceivingApplication = new Field();
        private Field _ReceivingFacility = new Field();
        private Field _DateTimeOfMessage = new Field();
        private Field _Security = new Field();
        private Field _MessageType = new Field();
        private Field _MessageControlID = new Field();
        private Field _ProcessingID = new Field();
        private Field _HL7Version = new Field();
        #endregion
        #region Constructors
        public MessageHeaderSegment(string message)
            : this() {
            //Has to at least be 8 chars for the type id (3 chars) and all the delimiter info (5 chars)
            if (message.Length < 8) {
                throw new ArgumentException("Message Header Segment is too short (MSH).  Expected at least 8 characters, returned " + message.Length);
            }
            SegmentTypeID.Value = message.Substring(0, 3);
            FieldDelimiter.Value = message.Substring(3, 1);
            ComponentDelimiter.Value = message.Substring(4, 1);
            RepeatDelimiter.Value = message.Substring(5, 1);
            EscapeCharacter.Value = message.Substring(6, 1);
            SubComponentDelimiter.Value = message.Substring(7, 1);
            //message.Substring(8) is the delimiter betweeh the delimiter section and the other fields
            List<string> fields = new List<string>(message.Substring(9).Split(this.FieldDelimiter.Value.ToCharArray(), StringSplitOptions.None));
            if (fields.Count < 10) {
                throw new ArgumentException("Not enough fields to construct a valid message header segment (MSH).  Expected at least 8, returned " + fields.Count);
            }
            SendingApplication.Value = fields[0];
            SendingFacility.Value = fields[1];
            ReceivingApplication.Value = fields[2];
            ReceivingFacility.Value = fields[3];
            DateTimeOfMessage.Value = fields[4];
            Security.Value = fields[5];
            MessageType.Value = fields[6];
            MessageControlID.Value = fields[7];
            ProcessingID.Value = fields[8];
            HL7Version.Value = fields[9];
            //Any additional segments that we don't know about
            if (fields.Count > 10) {
                for (int x = 10; x < fields.Count - 1; x++) {
                    Field newField = new Field("Additional unknown field", "MSH." + x, "Unknown", RequiredFlags.Unknown, -1, this.FieldDelimiter.Value, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Minus the first 6 and the last one
            foreach (Field messageField in this.KnownFields.GetRange(5, KnownFields.Count - 7)) {
                messageField.Delimiter = this.FieldDelimiter.ToString();
            }
            //Last delimiter always a newline
            HL7Version.Delimiter = System.Environment.NewLine;
        }
        public MessageHeaderSegment() {
            //Creates all the fields but without values, except for the delimiters 
            //Might be lab corp specific
            SegmentTypeID = new Field("Segment Type ID", "MSH.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, String.Empty, SupportedCharacters.AlphaOnly, 0, "MSH");
            FieldDelimiter = new Field("Field Delimiter", "MSH.2", "To seperate fields", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 1, string.Empty);
            ComponentDelimiter = new Field("Component Delimiter", "MSH.2", "To seperate components", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 2, string.Empty);
            RepeatDelimiter = new Field("Repeat Delimiter", "MSH.2", "To identify repeating fields", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 2, string.Empty);
            EscapeCharacter = new Field("Escape Character", "MSH.2", "To identify when a delimiter is used in the data as opposed to being used as a delimiter", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 2, string.Empty);
            SubComponentDelimiter = new Field("Sub-Component Delimiter", "MSH.2", "To identify sub-components", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 2, string.Empty);
            SendingApplication = new Field("Sending Application", "MSH.3", "Used to identify the system sending the results", RequiredFlags.AlwaysRequired, 4, String.Empty, SupportedCharacters.AlphaNumericSpecial, 3, string.Empty);
            SendingFacility = new Field("Sending Facility", "MSH.4", "Used to identify the facility sending the results", RequiredFlags.AlwaysRequired, 5, String.Empty, SupportedCharacters.AlphaNumericSpecial, 4, string.Empty);
            ReceivingApplication = new Field("Receiving Application", "MSH.5", "To identify the vendor for routing", RequiredFlags.AlwaysRequired, 10, String.Empty, SupportedCharacters.AlphaNumericSpecial, 5, string.Empty);
            ReceivingFacility = new Field("Receiving Facility", "MSH.6", "To identify the customer", RequiredFlags.AlwaysRequired, 10, String.Empty, SupportedCharacters.AlphaNumericSpecial, 6, string.Empty);
            DateTimeOfMessage = new Field("Date/Time of Message", "MSH.7", "Record of the date and time the electronic order was created", RequiredFlags.AlwaysRequired, 12, String.Empty, SupportedCharacters.NumericOnly, 7, string.Empty);
            Security = new Field("Security", "MSH.8", "Not currently used", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 8, string.Empty);
            MessageType = new Field("Message Type", "MSH.9", "To identify a result message being sent", RequiredFlags.AlwaysRequired, 3, String.Empty, SupportedCharacters.AlphaNumericSpecial, 9, string.Empty);
            MessageControlID = new Field("Message Control ID", "MSH.10", "Specimin sequence number", RequiredFlags.Optional, 4, String.Empty, SupportedCharacters.NumericOnly, 10, string.Empty);
            ProcessingID = new Field("Processing ID", "MSH.11", "To identify 'Production' order", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.AlphaNumericSpecial, 11, string.Empty);
            HL7Version = new Field("HL7 Version", "MSH.12", "Version of HL7 utilized for the format of this data structure", RequiredFlags.AlwaysRequired, 3, String.Empty, SupportedCharacters.AlphaNumericSpecial, 12, string.Empty);
        }
        public MessageHeaderSegment(string fieldDelimiter, string componentDelimiter, string repeatDelimiter, string escapeCharacter, string subComponentDelimiter)
            : this() {
            this.FieldDelimiter.Value = fieldDelimiter;
            this.ComponentDelimiter.Value = componentDelimiter;
            this.RepeatDelimiter.Value = repeatDelimiter;
            this.EscapeCharacter.Value = escapeCharacter;
            this.SubComponentDelimiter.Value = subComponentDelimiter;
            //Minus the first 6 and the last one
            foreach (Field messageField in this.KnownFields.GetRange(5, KnownFields.Count - 7)) {
                messageField.Delimiter = this.FieldDelimiter.ToString();
            }
            //Missed the last one, just do it here
            ProcessingID.Delimiter = this.FieldDelimiter.ToString();
            //Last delimiter always a newline
            HL7Version.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Instance Properties
        public Field FieldDelimiter {
            get {
                return _FieldDelimiter;
            }
            private set {
                _FieldDelimiter = value;
            }
        }
        public Field ComponentDelimiter {
            get {
                return _ComponentDelimiter;
            }
            private set {
                _ComponentDelimiter = value;
            }
        }
        public Field RepeatDelimiter {
            get {
                return _RepeatDelimiter;
            }
            private set {
                _RepeatDelimiter = value;
            }
        }
        public Field EscapeCharacter {
            get {
                return _EscapeCharacter;
            }
            private set {
                _EscapeCharacter = value;
            }
        }
        public Field SubComponentDelimiter {
            get {
                return _SubComponentDelimiter;
            }
            private set {
                _SubComponentDelimiter = value;
            }
        }
        public Field SendingApplication {
            get {
                return _SendingApplication;
            }
            private set {
                _SendingApplication = value;
            }
        }
        public Field SendingFacility {
            get {
                return _SendingFacility;
            }
            private set {
                _SendingFacility = value;
            }
        }
        public Field ReceivingApplication {
            get {
                return _ReceivingApplication;
            }
            private set {
                _ReceivingApplication = value;
            }
        }
        public Field ReceivingFacility {
            get {
                return _ReceivingFacility;
            }
            private set {
                _ReceivingFacility = value;
            }
        }
        public Field DateTimeOfMessage {
            get {
                return _DateTimeOfMessage;
            }
            private set {
                _DateTimeOfMessage = value;
            }
        }
        public DateTime? TypedDateTimeOfMessage {
            get {
                DateTime result;
                string format = DateTimeOfMessage.Value.Length == 12 ? "yyyyMMddHHmm" : "yyyyMMddHHmmss";
                bool parsed = DateTime.TryParseExact(DateTimeOfMessage.Value, format, System.Globalization.DateTimeFormatInfo.CurrentInfo, System.Globalization.DateTimeStyles.None, out result);
                if (parsed) {
                    return result;
                } else {
                    return null;
                } 
            }
        }
        public Field Security {
            get {
                return _Security;
            }
            private set {
                _Security = value;
            }
        }
        public Field MessageType {
            get {
                return _MessageType;
            }
            private set {
                _MessageType = value;
            }
        }
        public Field MessageControlID {
            get {
                return _MessageControlID;
            }
            private set {
                _MessageControlID = value;
            }
        }
        public Field ProcessingID {
            get {
                return _ProcessingID;
            }
            private set {
                _ProcessingID = value;
            }
        }
        public Field HL7Version {
            get {
                return _HL7Version;
            }
            private set {
                _HL7Version = value;
            }
        }
        public override List<Field> KnownFields {
            get {
                return new List<Field>(new Field[] { this.SegmentTypeID, this.FieldDelimiter, this.ComponentDelimiter, this.RepeatDelimiter, this.EscapeCharacter, this.SubComponentDelimiter, this.SendingApplication, this.SendingFacility, this.ReceivingApplication, this.ReceivingFacility, this.DateTimeOfMessage, this.Security, this.MessageType, this.MessageControlID, this.ProcessingID, this.HL7Version });
            }
        }
        public override string SegmentType {
            get {
                return "MSH";
            }
        }
        public override string Title {
            get {
                return "Message Header";
            }
        }
        #endregion
    }
}
