using System;
using System.Collections.Generic;
using System.Text;

namespace HL7Objects
{
    public enum SupportedCharacters
    {
        Unknown = 0,
        AlphaOnly = 1,
        NumericOnly = 2,
        SpecialOnly = 3,
        AlphaNumeric = 4,
        AlphaSpecial = 5,
        NumericSpecial = 6,
        AlphaNumericSpecial = 7,
        Unused = 8
    };
    public enum RequiredFlags
    {
        Unknown = 0,
        AlwaysRequired = 1,
        Conditional = 2,
        Optional = 3,
        Unused = 4
    };
    public enum RelationshipToPatient
    {
        Self = 1,
        Spouse = 2,
        Other = 3
    };
    public class Field
    {
        #region Private Instance Variables
        private string _Title;
        private string _Mnemonic;
        private string _Usage;
        private RequiredFlags _RequiredFlag = RequiredFlags.Unknown;
        private int _MaxLength = -1;
        private string _Delimiter;
        private SupportedCharacters _SupportedCharacters = SupportedCharacters.Unused;
        private int _HL7DataElement = -1;
        private string _Value;
        #endregion
        #region Constructors
        public Field(String title, string mnemonic, string usage, RequiredFlags requiredFlag, int maxLength, string fieldDelimiter, SupportedCharacters validCharacters, int hl7DataElement, string value)
            : this()
        {
            this.Title = title;
            this.Mnemonic = mnemonic;
            this.Usage = usage;
            this.RequiredFlag = requiredFlag;
            this.MaxLength = maxLength;
            this.Delimiter = fieldDelimiter;
            this._SupportedCharacters = validCharacters;
            this.HL7DataElement = hl7DataElement;
            this.Value = value;
        }
        public Field()
        {//Just to have a New()
        }
        #endregion
        #region Public Instance Properties
        public string Title
        {
            get { return _Title; }
            private set { _Title = value; }
        }
        public string Mnemonic
        {
            get { return _Mnemonic; }
            private set { _Mnemonic = value; }
        }
        public string Usage
        {
            get { return _Usage; }
            private set { _Usage = value; }
        }
        public RequiredFlags RequiredFlag
        {
            get { return _RequiredFlag; }
            private set { _RequiredFlag = value; }
        }
        public int MaxLength
        {
            get { return _MaxLength; }
            private set { _MaxLength = value; }
        }
        public string Delimiter
        {
            get { return _Delimiter; }
            internal set { _Delimiter = value; }
        }
        public string ValidCharacters
        {
            get { return GetSupportedCharacters(_SupportedCharacters); }

        }
        public int HL7DataElement
        {
            get { return _HL7DataElement; }
            private set { _HL7DataElement = value; }
        }
        public string Value
        {
            get { return _Value; }
            set { _Value = value; }
        }
        static string GetSupportedCharacters(SupportedCharacters supported)
        {
            switch (supported)
            {
                case SupportedCharacters.AlphaOnly:
                    return "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
                case SupportedCharacters.NumericOnly:
                    return "0123456789";
                case SupportedCharacters.SpecialOnly:
                    return @"~!@#$%^&*()_+{}|:<>?[]\;',./""";
                case SupportedCharacters.AlphaNumeric:
                    return GetSupportedCharacters(SupportedCharacters.AlphaOnly) + GetSupportedCharacters(SupportedCharacters.NumericOnly);
                case SupportedCharacters.AlphaSpecial:
                    return GetSupportedCharacters(SupportedCharacters.AlphaOnly) + GetSupportedCharacters(SupportedCharacters.SpecialOnly);
                case SupportedCharacters.NumericSpecial:
                    return GetSupportedCharacters(SupportedCharacters.NumericOnly) + GetSupportedCharacters(SupportedCharacters.SpecialOnly);
                case SupportedCharacters.AlphaNumericSpecial:
                    return GetSupportedCharacters(SupportedCharacters.AlphaOnly) + GetSupportedCharacters(SupportedCharacters.NumericOnly) + GetSupportedCharacters(SupportedCharacters.SpecialOnly);
                case SupportedCharacters.Unused:
                    return string.Empty;
            }
            return string.Empty;
        }
        public string RightSideDelimitedValue
        {
            get { return this.Value + this.Delimiter; }
        }
        #endregion
        #region Public Instance Methods
        public override string ToString()
        {
            return this.Value;
        }
        #endregion
    }
    public class MessageHeaderSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _FieldDelimiter = new Field();
        private Field _ComponentDelimiter = new Field();
        private Field _RepeatDelimiter = new Field();
        private Field _EscapeCharacter = new Field();
        private Field _SubComponentDelimiter = new Field();
        private Field _SendingApplication = new Field();
        private Field _SendingFacility = new Field();
        private Field _ReceivingApplication = new Field();
        private Field _ReceivingFacility = new Field();
        private Field _DateTimeOfMessage = new Field();
        private Field _Security = new Field();
        private Field _MessageType = new Field();
        private Field _MessageControlID = new Field();
        private Field _ProcessingID = new Field();
        private Field _HL7Version = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Constructors
        public MessageHeaderSegment(string message)
            : this()
        {
            //Has to at least be 8 chars for the type id (3 chars) and all the delimiter info (5 chars)
            if (message.Length < 8) { throw new ArgumentException("Message Header Segment is too short (MSH).  Expected at least 8 characters, returned " + message.Length); }
            SegmentTypeID.Value = message.Substring(0, 3);
            FieldDelimiter.Value = message.Substring(3, 1);
            ComponentDelimiter.Value = message.Substring(4, 1);
            RepeatDelimiter.Value = message.Substring(5, 1);
            EscapeCharacter.Value = message.Substring(6, 1);
            SubComponentDelimiter.Value = message.Substring(7, 1);
            //message.Substring(8) is the delimiter betweeh the delimiter section and the other fields
            List<string> fields = new List<string>(message.Substring(9).Split(this.FieldDelimiter.Value.ToCharArray(), StringSplitOptions.None));
            if (fields.Count < 10) { throw new ArgumentException("Not enough fields to construct a valid message header segment (MSH).  Expected at least 8, returned " + fields.Count); }
            SendingApplication.Value = fields[0];
            SendingFacility.Value = fields[1];
            ReceivingApplication.Value = fields[2];
            ReceivingFacility.Value = fields[3];
            DateTimeOfMessage.Value = fields[4];
            Security.Value = fields[5];
            MessageType.Value = fields[6];
            MessageControlID.Value = fields[7];
            ProcessingID.Value = fields[8];
            HL7Version.Value = fields[9];
            //Any additional segments that we don't know about
            if (fields.Count > 10)
            {
                for (int x = 10; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "MSH." + x, "Unknown", RequiredFlags.Unknown, -1, this.FieldDelimiter.Value, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            foreach (Field messageField in this.KnownFields.GetRange(5, KnownFields.Count - 7))//Minus the first 6 and the last one
            {
                messageField.Delimiter = this.FieldDelimiter.ToString();
            }
            //Last delimiter always a newline
            HL7Version.Delimiter = System.Environment.NewLine;
        }
        public MessageHeaderSegment()
        {
            //Creates all the segments but without values. 
            //Might be lab corp specific
            SegmentTypeID = new Field("Segment Type ID", "MSH.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, String.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            FieldDelimiter = new Field("Field Delimiter", "MSH.2", "To seperate fields", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 1, string.Empty);
            ComponentDelimiter = new Field("Component Delimiter", "MSH.2", "To seperate components", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 2, string.Empty);
            RepeatDelimiter = new Field("Repeat Delimiter", "MSH.2", "To identify repeating fields", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 2, string.Empty);
            EscapeCharacter = new Field("Escape Character", "MSH.2", "To identify when a delimiter is used in the data as opposed to being used as a delimiter", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 2, string.Empty);
            SubComponentDelimiter = new Field("Sub-Component Delimiter", "MSH.2", "To identify sub-components", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.SpecialOnly, 2, string.Empty);
            SendingApplication = new Field("Sending Application", "MSH.3", "Used to identify the system sending the results", RequiredFlags.AlwaysRequired, 4, String.Empty, SupportedCharacters.AlphaNumericSpecial, 3, string.Empty);
            SendingFacility = new Field("Sending Facility", "MSH.4", "Used to identify the facility sending the results", RequiredFlags.AlwaysRequired, 5, String.Empty, SupportedCharacters.AlphaNumericSpecial, 4, string.Empty);
            ReceivingApplication = new Field("Receiving Application", "MSH.5", "To identify the vendor for routing", RequiredFlags.AlwaysRequired, 10, String.Empty, SupportedCharacters.AlphaNumericSpecial, 5, string.Empty);
            ReceivingFacility = new Field("Receiving Facility", "MSH.6", "To identify the customer", RequiredFlags.AlwaysRequired, 10, String.Empty, SupportedCharacters.AlphaNumericSpecial, 6, string.Empty);
            DateTimeOfMessage = new Field("Date/Time of Message", "MSH.7", "Record of the date and time the electronic order was created", RequiredFlags.AlwaysRequired, 12, String.Empty, SupportedCharacters.NumericOnly, 7, string.Empty);
            Security = new Field("Security", "MSH.8", "Not currently used", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 8, string.Empty);
            MessageType = new Field("Message Type", "MSH.9", "To identify a result message being sent", RequiredFlags.AlwaysRequired, 3, String.Empty, SupportedCharacters.AlphaNumericSpecial, 9, string.Empty);
            MessageControlID = new Field("Message Control ID", "MSH.10", "Specimin sequence number", RequiredFlags.Optional, 4, String.Empty, SupportedCharacters.NumericOnly, 10, string.Empty);
            ProcessingID = new Field("Processing ID", "MSH.11", "To identify 'Production' order", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.AlphaNumericSpecial, 11, string.Empty);
            HL7Version = new Field("HL7 Version", "MSH.12", "Version of HL7 utilized for the format of this data structure", RequiredFlags.AlwaysRequired, 3, String.Empty, SupportedCharacters.AlphaNumericSpecial, 12, string.Empty);
        }
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID
        {
            get { return _SegmentTypeID; }
            protected set { _SegmentTypeID = value; }
        }
        public Field FieldDelimiter
        {
            get { return _FieldDelimiter; }
            private set { _FieldDelimiter = value; }
        }
        public Field ComponentDelimiter
        {
            get { return _ComponentDelimiter; }
            private set { _ComponentDelimiter = value; }
        }
        public Field RepeatDelimiter
        {
            get { return _RepeatDelimiter; }
            private set { _RepeatDelimiter = value; }
        }
        public Field EscapeCharacter
        {
            get { return _EscapeCharacter; }
            private set { _EscapeCharacter = value; }
        }
        public Field SubComponentDelimiter
        {
            get { return _SubComponentDelimiter; }
            private set { _SubComponentDelimiter = value; }
        }
        public Field SendingApplication
        {
            get { return _SendingApplication; }
            private set { _SendingApplication = value; }
        }
        public Field SendingFacility
        {
            get { return _SendingFacility; }
            private set { _SendingFacility = value; }
        }
        public Field ReceivingApplication
        {
            get { return _ReceivingApplication; }
            private set { _ReceivingApplication = value; }
        }
        public Field ReceivingFacility
        {
            get { return _ReceivingFacility; }
            private set { _ReceivingFacility = value; }
        }
        public Field DateTimeOfMessage
        {
            get { return _DateTimeOfMessage; }
            private set { _DateTimeOfMessage = value; }
        }
        public DateTime TypedDateTimeOfMessage
        {
            get { return DateTime.Parse(this.DateTimeOfMessage.Value); }
        }
        public Field Security
        {
            get { return _Security; }
            private set { _Security = value; }
        }
        public Field MessageType
        {
            get { return _MessageType; }
            private set { _MessageType = value; }
        }
        public Field MessageControlID
        {
            get { return _MessageControlID; }
            private set { _MessageControlID = value; }
        }
        public Field ProcessingID
        {
            get { return _ProcessingID; }
            private set { _ProcessingID = value; }
        }
        public Field HL7Version
        {
            get { return _HL7Version; }
            private set { _HL7Version = value; }
        }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { this.SegmentTypeID, this.FieldDelimiter, this.ComponentDelimiter, this.RepeatDelimiter, this.EscapeCharacter, this.SubComponentDelimiter, this.SendingApplication, this.SendingFacility, this.ReceivingApplication, this.ReceivingFacility, this.DateTimeOfMessage, this.Security, this.MessageType, this.MessageControlID, this.ProcessingID, this.HL7Version }); }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field messageField in this.AllFields)
                {
                    builder.Append(messageField.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field messageField in this.KnownFields)
                {
                    builder.Append(messageField.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "MSH"; }
        }
        #endregion
    }
    public class PatientIdentificationSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _ExternalPatientID = new Field();
        private Field _LabAssignedPatientID = new Field();
        private Field _AlternatePatientID = new Field();
        private Field _PatientLastName = new Field();
        private Field _PatientFirstName = new Field();
        private Field _PatientMiddleInitial = new Field();
        private Field _MothersMaidenName = new Field();
        private Field _PatientDateOfBirth = new Field();
        private Field _PatientSex = new Field();
        private Field _PatientAlias = new Field();
        private Field _PatientRace = new Field();
        private Field _PatientAddress = new Field();
        private Field _PatientOtherDesignation = new Field();
        private Field _PatientCity = new Field();
        private Field _PatientStateOrProvice = new Field();
        private Field _PatientZipOrPostalCode = new Field();
        private Field _PatientCountryCode = new Field();
        private Field _PatientHomePhoneNumber = new Field();
        private Field _PatientWorkPhoneNumber = new Field();
        private Field _PatientLanguage = new Field();
        private Field _PatientMaritalStatus = new Field();
        private Field _PatientReligion = new Field();
        private Field _CustomerID = new Field();
        private Field _CheckDigit = new Field();
        private Field _CheckDigitScheme = new Field();
        private Field _BillCode = new Field();
        private Field _ABNFlag = new Field();
        private Field _StatusOfSpecimen = new Field();
        private Field _Fasting = new Field();
        private Field _PatientSSNNumber = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Constructors
        public PatientIdentificationSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 20) { throw new ArgumentException("Not enough fields to construct a valid patient identification segment (PID).  Expected at least 20, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.SequenceNumber.Value = fields[1];
            this.ExternalPatientID.Value = fields[2];
            this.LabAssignedPatientID.Value = fields[3];
            this.AlternatePatientID.Value = fields[4];
            if (fields[5].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'PID.6'"); }
            List<string> components = new List<string>(fields[5].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count < 2) { throw new ArgumentException("Unexpected number of components returned for the field 'PID.6'.  Expected 2, returned " + components.Count); }
            this.PatientLastName.Value = components[0];
            this.PatientFirstName.Value = components[1];
            if (components.Count > 2) { this.PatientMiddleInitial.Value = components[2]; }//Might not be there
            //Back to non-nested
            this.MothersMaidenName.Value = fields[6];
            this.PatientDateOfBirth.Value = fields[7];
            this.PatientSex.Value = fields[8];
            this.PatientAlias.Value = fields[9];
            this.PatientRace.Value = fields[10];
            //Another group of nested fields
            if (fields[11].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'PID.12'"); }
            components = new List<string>(fields[11].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            if (components.Count != 5) { throw new ArgumentException("Unexpected number of components returned for the field 'PID.12'.  Expected 5, returned " + components.Count); }
            this.PatientAddress.Value = components[0];
            this.PatientOtherDesignation.Value = components[1];
            this.PatientCity.Value = components[2];
            this.PatientStateOrProvice.Value = components[3];
            this.PatientZipOrPostalCode.Value = components[4];
            //Back to non-nested again
            this.PatientCountryCode.Value = fields[12];
            this.PatientHomePhoneNumber.Value = fields[13];
            this.PatientWorkPhoneNumber.Value = fields[14];
            this.PatientLanguage.Value = fields[15];
            this.PatientMaritalStatus.Value = fields[16];
            this.PatientReligion.Value = fields[17];
            //More nested
            //Status of specimen and Fasting is not used in the Order Message (ORM) and ABN flag is optional, why the count is 4 and not 7
            if (fields[18].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'PID.19'"); }
            components = new List<string>(fields[18].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            if (components.Count < 4) { throw new ArgumentException("Unexpected number of components returned for the field 'PID.19'.  Expected 4, returned " + components.Count); }
            this.CustomerID.Value = components[0];
            this.CheckDigit.Value = components[1];
            this.CheckDigitScheme.Value = components[2];
            this.BillCode.Value = components[3];
            if (components.Count >= 5) { this.ABNFlag.Value = components[4]; }//Optional in the ORM
            if (components.Count >= 6) { this.StatusOfSpecimen.Value = components[5]; }//Not used in the Order Message (ORM), just in the Result Message (ORU)
            if (components.Count >= 6) { this.Fasting.Value = components[6]; }//Not used in the Order Message (ORM), just in the Result Message (ORU)
            //Last one non-nested
            this.PatientSSNNumber.Value = fields[19];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 20)
            {
                for (int x = 20; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "PID." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Non-nested field delimiters
            foreach (Field patientIdentification in KnownFields)
            {
                patientIdentification.Delimiter = fieldDelimiter;
            }
            //Nested field delimiters
            foreach (Field patientIdentification in new Field[] { PatientLastName, PatientFirstName, PatientAddress, PatientOtherDesignation, PatientCity, PatientStateOrProvice, CustomerID, CheckDigit, CheckDigitScheme, BillCode, ABNFlag, StatusOfSpecimen })
            {
                patientIdentification.Delimiter = componentDelimiter;
            }
            //last delimiter is always a newline
            this.PatientSSNNumber.Delimiter = System.Environment.NewLine;
        }
        public PatientIdentificationSegment()
        {   //Creates all the segments but without values. 
            //Might be lab corp specific
            SegmentTypeID = new Field("Segment Type ID", "PID.1", "Used to identify the segment", RequiredFlags.AlwaysRequired, 3, String.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            SequenceNumber = new Field("Sequence Number", "PID.2", "To identify the number of PID segments contained within the order file", RequiredFlags.AlwaysRequired, 4, String.Empty, SupportedCharacters.NumericOnly, 104, string.Empty);
            ExternalPatientID = new Field("External Patient ID", "PID.3", "Customer assigned patient identifier", RequiredFlags.AlwaysRequired, 20, String.Empty, SupportedCharacters.AlphaNumericSpecial, 105, string.Empty);
            LabAssignedPatientID = new Field("Lab Assigned Patient ID", "PID.4", "Assigned specimen number", RequiredFlags.AlwaysRequired, 11, String.Empty, SupportedCharacters.AlphaNumericSpecial, 106, string.Empty);
            AlternatePatientID = new Field("Alternate Patient ID", "PID.5", "Customer assigned patient identifier", RequiredFlags.Optional, 20, String.Empty, SupportedCharacters.AlphaNumericSpecial, 107, string.Empty);
            PatientLastName = new Field("Patient Last Name", "PID.6", "Patient Demographics", RequiredFlags.AlwaysRequired, 12, String.Empty, SupportedCharacters.AlphaOnly, 108, string.Empty);
            PatientFirstName = new Field("Patient First Name", "PID.6", "Patient Demographics", RequiredFlags.AlwaysRequired, 10, String.Empty, SupportedCharacters.AlphaOnly, 108, string.Empty);
            PatientMiddleInitial = new Field("Patient Middle Initial", "PID.6", "Patient Demographics", RequiredFlags.Optional, 1, String.Empty, SupportedCharacters.AlphaOnly, 108, string.Empty);
            MothersMaidenName = new Field("Mothers Maiden Name", "PID.7", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 109, string.Empty);
            PatientDateOfBirth = new Field("Patient Date of Birth", "PID.8", "Patient Demographics", RequiredFlags.AlwaysRequired, 8, String.Empty, SupportedCharacters.NumericOnly, 110, string.Empty);
            PatientSex = new Field("Patient Sex", "PID.9", "Patient Demographics", RequiredFlags.Optional, 1, String.Empty, SupportedCharacters.AlphaOnly, 111, string.Empty);
            PatientAlias = new Field("Patient Alias", "PID.10", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 112, string.Empty);
            PatientRace = new Field("Patient Race", "PID.11", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 113, string.Empty);
            PatientAddress = new Field("Patient Address", "PID.12", "Patient Demographics", RequiredFlags.Optional, 35, String.Empty, SupportedCharacters.AlphaNumericSpecial, 114, string.Empty);
            PatientOtherDesignation = new Field("Patient Other Designation", "PID.12", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 114, string.Empty);
            PatientCity = new Field("Patient City", "PID.12", "Patient Demographics", RequiredFlags.Optional, 16, String.Empty, SupportedCharacters.AlphaOnly, 114, string.Empty);
            PatientStateOrProvice = new Field("Patient State or Provice", "PID.12", "Patient Demographics", RequiredFlags.Optional, 2, String.Empty, SupportedCharacters.AlphaOnly, 114, string.Empty);
            PatientZipOrPostalCode = new Field("Patient Zip or PostalCode", "PID.12", "Patient Demographics", RequiredFlags.Optional, 9, String.Empty, SupportedCharacters.NumericOnly, 114, string.Empty);
            PatientCountryCode = new Field("Patient Country Code", "PID.13", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 115, string.Empty);
            PatientHomePhoneNumber = new Field("Patient Home Phone Number", "PID.14", "Patient Demographics", RequiredFlags.Optional, 13, String.Empty, SupportedCharacters.NumericSpecial, 116, string.Empty);
            PatientWorkPhoneNumber = new Field("Patient Work Phone Number", "PID.15", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 117, string.Empty);
            PatientLanguage = new Field("Patient Language", "PID.16", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 118, string.Empty);
            PatientMaritalStatus = new Field("Patient Marital Status", "PID.17", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 119, string.Empty);
            PatientReligion = new Field("Patient Religion", "PID.18", "Patient Demographics", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 120, string.Empty);
            CustomerID = new Field("Customer ID", "PID.19", "Customer Identification", RequiredFlags.AlwaysRequired, 8, String.Empty, SupportedCharacters.NumericOnly, 121, string.Empty);
            CheckDigit = new Field("Check Digit", "PID.19", "Unknown", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 121, string.Empty);
            CheckDigitScheme = new Field("Check Digit Scheme", "PID.19", "Unknown", RequiredFlags.Unused, -1, String.Empty, SupportedCharacters.Unused, 121, string.Empty);
            BillCode = new Field("Bill Code", "PID.19", "Billing Information", RequiredFlags.AlwaysRequired, 2, String.Empty, SupportedCharacters.AlphaNumeric, 121, string.Empty);
            ABNFlag = new Field("ABN Flag", "PID.19", "To determine if patient wavier was signed", RequiredFlags.Optional, 1, String.Empty, SupportedCharacters.AlphaOnly, 121, string.Empty);
            StatusOfSpecimen = new Field("Status of Specimen", "PID.19", "Indication of preliminary or final report", RequiredFlags.AlwaysRequired, 1, String.Empty, SupportedCharacters.AlphaOnly, 121, string.Empty);
            Fasting = new Field("Fasting", "PID.19", "Patient Demographics", RequiredFlags.Optional, 1, String.Empty, SupportedCharacters.AlphaOnly, 121, string.Empty);
            PatientSSNNumber = new Field("Patient SSN Number", "PID.20", "Patient Demographics", RequiredFlags.Optional, 9, String.Empty, SupportedCharacters.NumericOnly, 122, string.Empty);
        }
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID
        {
            get { return _SegmentTypeID; }
            private set { _SegmentTypeID = value; }
        }
        public Field SequenceNumber
        {
            get { return _SequenceNumber; }
            private set { _SequenceNumber = value; }
        }
        public Field ExternalPatientID
        {
            get { return _ExternalPatientID; }
            private set { _ExternalPatientID = value; }
        }
        public Field LabAssignedPatientID
        {
            get { return _LabAssignedPatientID; }
            private set { _LabAssignedPatientID = value; }
        }
        public Field AlternatePatientID
        {
            get { return _AlternatePatientID; }
            private set { _AlternatePatientID = value; }
        }
        public Field PatientLastName
        {
            get { return _PatientLastName; }
            private set { _PatientLastName = value; }
        }
        public Field PatientFirstName
        {
            get { return _PatientFirstName; }
            private set { _PatientFirstName = value; }
        }
        public Field PatientMiddleInitial
        {
            get { return _PatientMiddleInitial; }
            private set { _PatientMiddleInitial = value; }
        }
        public Field MothersMaidenName
        {
            get { return _MothersMaidenName; }
            private set { _MothersMaidenName = value; }
        }
        public Field PatientDateOfBirth
        {
            get { return _PatientDateOfBirth; }
            private set { _PatientDateOfBirth = value; }
        }
        public DateTime TypedPatientDateOfBirth
        {
            get { return DateTime.Parse(this.PatientDateOfBirth.Value); }
        }
        public Field PatientSex
        {
            get { return _PatientSex; }
            private set { _PatientSex = value; }
        }
        public Field PatientAlias
        {
            get { return _PatientAlias; }
            private set { _PatientAlias = value; }
        }
        public Field PatientRace
        {
            get { return _PatientRace; }
            private set { _PatientRace = value; }
        }
        public Field PatientAddress
        {
            get { return _PatientAddress; }
            private set { _PatientAddress = value; }
        }
        public Field PatientOtherDesignation
        {
            get { return _PatientOtherDesignation; }
            private set { _PatientOtherDesignation = value; }
        }
        public Field PatientCity
        {
            get { return _PatientCity; }
            private set { _PatientCity = value; }
        }
        public Field PatientStateOrProvice
        {
            get { return _PatientStateOrProvice; }
            private set { _PatientStateOrProvice = value; }
        }
        public Field PatientZipOrPostalCode
        {
            get { return _PatientZipOrPostalCode; }
            private set { _PatientZipOrPostalCode = value; }
        }
        public Field PatientCountryCode
        {
            get { return _PatientCountryCode; }
            private set { _PatientCountryCode = value; }
        }
        public Field PatientHomePhoneNumber
        {
            get { return _PatientHomePhoneNumber; }
            private set { _PatientHomePhoneNumber = value; }
        }
        public Field PatientWorkPhoneNumber
        {
            get { return _PatientWorkPhoneNumber; }
            private set { _PatientWorkPhoneNumber = value; }
        }
        public Field PatientLanguage
        {
            get { return _PatientLanguage; }
            private set { _PatientLanguage = value; }
        }
        public Field PatientMaritalStatus
        {
            get { return _PatientMaritalStatus; }
            private set { _PatientMaritalStatus = value; }
        }
        public Field PatientReligion
        {
            get { return _PatientReligion; }
            private set { _PatientReligion = value; }
        }
        public Field CustomerID
        {
            get { return _CustomerID; }
            private set { _CustomerID = value; }
        }
        public Field CheckDigit
        {
            get { return _CheckDigit; }
            private set { _CheckDigit = value; }
        }
        public Field CheckDigitScheme
        {
            get { return _CheckDigitScheme; }
            private set { _CheckDigitScheme = value; }
        }
        public Field BillCode
        {
            get { return _BillCode; }
            private set { _BillCode = value; }
        }
        public Field ABNFlag
        {
            get { return _ABNFlag; }
            private set { _ABNFlag = value; }
        }
        public Field StatusOfSpecimen
        {
            get { return _StatusOfSpecimen; }
            private set { _StatusOfSpecimen = value; }
        }
        public Field Fasting
        {
            get { return _Fasting; }
            private set { _Fasting = value; }
        }
        public Field PatientSSNNumber
        {
            get { return _PatientSSNNumber; }
            private set { _PatientSSNNumber = value; }
        }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { this.SegmentTypeID, this.SequenceNumber, this.ExternalPatientID, this.LabAssignedPatientID, this.AlternatePatientID, this.PatientLastName, this.PatientFirstName, this.PatientMiddleInitial, this.MothersMaidenName, this.PatientDateOfBirth, this.PatientSex, this.PatientAlias, this.PatientCity, this.PatientStateOrProvice, this.PatientZipOrPostalCode, this.PatientCountryCode, this.PatientHomePhoneNumber, this.PatientWorkPhoneNumber, this.PatientLanguage, this.PatientMaritalStatus, this.PatientReligion, this.CustomerID, this.CheckDigit, this.CheckDigitScheme, this.BillCode, this.ABNFlag, this.StatusOfSpecimen, this.Fasting, this.PatientSSNNumber }); }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field patientIdentification in this.AllFields)
                {
                    builder.Append(patientIdentification.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field patientIdentification in this.KnownFields)
                {
                    builder.Append(patientIdentification.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "PID"; }
        }
        #endregion
    }
    public class CommonOrderSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _OrderControl = new Field();
        private Field _UniqueForeignAccessionOrSpecimenID = new Field();
        private Field _ApplicationInstitutionID = new Field();
        private Field _FillerAccessionID = new Field();
        private Field _PlacerGroupNumber = new Field();
        private Field _OrderStatus = new Field();
        private Field _ResponseFlag = new Field();
        private Field _QuantityTiming = new Field();
        private Field _Parent = new Field();
        private Field _DateTimeOfTransaction = new Field();
        private Field _EnteredBy = new Field();
        private Field _VerifiedBy = new Field();
        private Field _OrderingProviderIDNumber = new Field();
        private Field _OrderingProviderLastName = new Field();
        private Field _OrderingProviderFirstInitial = new Field();
        private Field _OrderingProviderMiddleInitial = new Field();
        private Field _OrderingProviderSuffix = new Field();
        private Field _OrderingProviderPrefix = new Field();
        private Field _OrderingProviderDegree = new Field();
        private Field _SourceTable = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID
        {
            get { return _SegmentTypeID; }
            private set { _SegmentTypeID = value; }
        }
        public Field OrderControl
        {
            get { return _OrderControl; }
            private set { _OrderControl = value; }
        }
        public Field UniqueForeignAccessionOrSpecimenID
        {
            get { return _UniqueForeignAccessionOrSpecimenID; }
            private set { _UniqueForeignAccessionOrSpecimenID = value; }
        }
        public Field ApplicationInstitutionID
        {
            get { return _ApplicationInstitutionID; }
            private set { _ApplicationInstitutionID = value; }
        }
        public Field FillerAccessionID
        {
            get { return _FillerAccessionID; }
            private set { _FillerAccessionID = value; }
        }
        public Field PlacerGroupNumber
        {
            get { return _PlacerGroupNumber; }
            private set { _PlacerGroupNumber = value; }
        }
        public Field OrderStatus
        {
            get { return _OrderStatus; }
            private set { _OrderStatus = value; }
        }
        public Field ResponseFlag
        {
            get { return _ResponseFlag; }
            private set { _ResponseFlag = value; }
        }
        public Field QuantityTiming
        {
            get { return _QuantityTiming; }
            private set { _QuantityTiming = value; }
        }
        public Field Parent
        {
            get { return _Parent; }
            private set { _Parent = value; }
        }
        public Field DateTimeOfTransaction
        {
            get { return _DateTimeOfTransaction; }
            private set { _DateTimeOfTransaction = value; }
        }
        public DateTime TypedDateTimeOfTransaction
        {
            get { return DateTime.Parse(DateTimeOfTransaction.Value); }
        }
        public Field EnteredBy
        {
            get { return _EnteredBy; }
            private set { _EnteredBy = value; }
        }
        public Field VerifiedBy
        {
            get { return _VerifiedBy; }
            private set { _VerifiedBy = value; }
        }
        public Field OrderingProviderIDNumber
        {
            get { return _OrderingProviderIDNumber; }
            private set { _OrderingProviderIDNumber = value; }
        }
        public Field OrderingProviderLastName
        {
            get { return _OrderingProviderLastName; }
            private set { _OrderingProviderLastName = value; }
        }
        public Field OrderingProviderFirstInitial
        {
            get { return _OrderingProviderFirstInitial; }
            private set { _OrderingProviderFirstInitial = value; }
        }
        public Field OrderingProviderMiddleInitial
        {
            get { return _OrderingProviderMiddleInitial; }
            private set { _OrderingProviderMiddleInitial = value; }
        }
        public Field OrderingProviderSuffix
        {
            get { return _OrderingProviderSuffix; }
            private set { _OrderingProviderSuffix = value; }
        }
        public Field OrderingProviderPrefix
        {
            get { return _OrderingProviderPrefix; }
            private set { _OrderingProviderPrefix = value; }
        }
        public Field OrderingProviderDegree
        {
            get { return _OrderingProviderDegree; }
            private set { _OrderingProviderDegree = value; }
        }
        public Field SourceTable
        {
            get { return _SourceTable; }
            private set { _SourceTable = value; }
        }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, OrderControl, UniqueForeignAccessionOrSpecimenID, ApplicationInstitutionID, FillerAccessionID, PlacerGroupNumber, OrderStatus, ResponseFlag, QuantityTiming, Parent, DateTimeOfTransaction, EnteredBy, VerifiedBy, OrderingProviderIDNumber, OrderingProviderLastName, OrderingProviderFirstInitial, OrderingProviderMiddleInitial, OrderingProviderSuffix, OrderingProviderPrefix, OrderingProviderDegree, SourceTable }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public CommonOrderSegment()
        {
            this.SegmentTypeID = new Field("Segment Type ID", "ORC.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.OrderControl = new Field("Order Control", "ORC.2", "To identify new orders", RequiredFlags.AlwaysRequired, 2, string.Empty, SupportedCharacters.AlphaOnly, 215, string.Empty);
            this.UniqueForeignAccessionOrSpecimenID = new Field("Unique Foreign Accession orSpecimen ID", "ORC.3", "Customer specific accessioning or specimen ID", RequiredFlags.AlwaysRequired, 25, string.Empty, SupportedCharacters.AlphaNumericSpecial, 216, string.Empty);
            this.ApplicationInstitutionID = new Field("Application Institution ID", "ORC.3", "Customer specific accessioning or specimen ID", RequiredFlags.AlwaysRequired, 6, string.Empty, SupportedCharacters.AlphaNumericSpecial, 216, string.Empty);
            this.FillerAccessionID = new Field("Filler Accession ID", "ORC.4", "Used in outbound file only", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 217, string.Empty);
            this.PlacerGroupNumber = new Field("Placer Group Number", "ORC.5", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 218, string.Empty);
            this.OrderStatus = new Field("Order Status", "ORC.6", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 219, string.Empty);
            this.ResponseFlag = new Field("Response Flag", "ORC.7", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 220, string.Empty);
            this.QuantityTiming = new Field("Quantity Timing", "ORC.8", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 221, string.Empty);
            this.Parent = new Field("Parent", "ORC.9", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 222, string.Empty);
            this.DateTimeOfTransaction = new Field("Date/Time of Transaction", "ORC.10", "Record of the date and time the electronic order was created", RequiredFlags.AlwaysRequired, 14, string.Empty, SupportedCharacters.NumericOnly, 223, string.Empty);
            this.EnteredBy = new Field("Entered By", "ORC.11", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 224, string.Empty);
            this.VerifiedBy = new Field("Verified By", "ORC.12", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 225, string.Empty);
            this.OrderingProviderIDNumber = new Field("Ordering Provider ID Number", "ORC.13", "Physcian/provider ID and required for certain billing options", RequiredFlags.Optional, 20, string.Empty, SupportedCharacters.AlphaNumericSpecial, 226, string.Empty);
            this.OrderingProviderLastName = new Field("Ordering Provider Last Name", "ORC.13", "Physcian/provider ID and required for certain billing options", RequiredFlags.Optional, 9, string.Empty, SupportedCharacters.AlphaOnly, 226, string.Empty);
            this.OrderingProviderFirstInitial = new Field("Ordering Provider First Initial", "ORC.13", "Physcian/provider ID and required for certain billing options", RequiredFlags.Optional, 1, string.Empty, SupportedCharacters.AlphaOnly, 226, string.Empty);
            this.OrderingProviderMiddleInitial = new Field("Ordering Provider Middle Initial", "ORC.13", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 226, string.Empty);
            this.OrderingProviderSuffix = new Field("Ordering Provider Suffix", "ORC.13", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 226, string.Empty);
            this.OrderingProviderPrefix = new Field("Ordering Provider Prefix", "ORC.13", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 226, string.Empty);
            this.OrderingProviderDegree = new Field("Ordering Provider Degree", "ORC.13", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 226, string.Empty);
            this.SourceTable = new Field("Source Table", "ORC.13", "Identification of ordering provider ID sent in ORC 13.1", RequiredFlags.Conditional, 1, string.Empty, SupportedCharacters.AlphaOnly, 226, string.Empty);
        }
        public CommonOrderSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 13) { throw new ArgumentException("Not enough fields to construct a valid common order segment (ORC).  Expected at least 13, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.OrderControl.Value = fields[1];
            if (fields[2].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ORC.3'"); }
            List<string> components = new List<string>(fields[2].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned from field 'ORC.3'.  Expected 2, returned " + components.Count); }
            this.UniqueForeignAccessionOrSpecimenID.Value = components[0];
            this.ApplicationInstitutionID.Value = components[1];
            //Back to non-nested fields
            this.FillerAccessionID.Value = fields[3];
            this.PlacerGroupNumber.Value = fields[4];
            this.OrderStatus.Value = fields[5];
            this.ResponseFlag.Value = fields[6];
            this.QuantityTiming.Value = fields[7];
            this.Parent.Value = fields[8];
            this.DateTimeOfTransaction.Value = fields[9];
            this.EnteredBy.Value = fields[10];
            this.VerifiedBy.Value = fields[11];
            if (fields[12].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ORC.13'"); }
            components = new List<string>(fields[12].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 8) { throw new ArgumentException("Unexpected number of components returned from field 'ORC.13'.  Expected 8, returned " + components.Count); }
            this.OrderingProviderIDNumber.Value = components[0];
            this.OrderingProviderLastName.Value = components[1];
            this.OrderingProviderFirstInitial.Value = components[2];
            this.OrderingProviderMiddleInitial.Value = components[3];
            this.OrderingProviderSuffix.Value = components[4];
            this.OrderingProviderPrefix.Value = components[5];
            this.OrderingProviderDegree.Value = components[6];
            this.SourceTable.Value = components[7];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 13)
            {
                for (int x = 13; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "ORC." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiters
            foreach (Field commonOrder in KnownFields)
            {
                commonOrder.Delimiter = fieldDelimiter;
            }
            //Comonent (nested fields) Delimiters
            foreach (Field commonOrder in new Field[] { UniqueForeignAccessionOrSpecimenID, OrderingProviderIDNumber, OrderingProviderLastName, OrderingProviderFirstInitial, OrderingProviderMiddleInitial, OrderingProviderSuffix, OrderingProviderPrefix, OrderingProviderDegree })
            {
                commonOrder.Delimiter = componentDelimiter;
            }
            //Last field is always a newline
            SourceTable.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "ORC"; }
        }
        #endregion
    }
    public class ObservationOrderSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _UniqueForeignSpecimenID = new Field();
        private Field _ForeignApplicationID = new Field();
        private Field _InternalSpecimenID = new Field();
        private Field _InternalApplicationID = new Field();
        private Field _ObservationBatteryID = new Field();
        private Field _ObservationBatteryText = new Field();
        private Field _NameOfCodingSystem = new Field();
        private Field _Priority = new Field();
        private Field _RequestedDateTime = new Field();
        private Field _ObservationCollectionDateTime = new Field();
        private Field _ObservationCollectionEndDateTime = new Field();
        private Field _CollectionVolume = new Field();
        private Field _CollectorIdentifier = new Field();
        private Field _SpecimenActionCode = new Field();
        private Field _DangerCode = new Field();
        private Field _UnknownField = new Field();
        private Field _RelevantClinicalInformation = new Field();
        private Field _SpecimenReceivedDateTime = new Field();
        private Field _SpecimenSource = new Field();
        private Field _OrderingProvider = new Field();
        private Field _OrderingProviderPhoneNumber = new Field();
        private Field _AlternateUniqueForeignSpecimenID = new Field();
        private Field _RequestorField2 = new Field();
        private Field _ProducerField1 = new Field();
        private Field _ProducerField2 = new Field();
        private Field _DateTimeObservationReported = new Field();
        private Field _ProducerCharge = new Field();
        private Field _ProducerSectionID = new Field();
        private Field _OrderResultStatus = new Field();
        private Field _LinkToParentResult = new Field();
        private Field _QuantityTiming = new Field();
        private Field _ResultCopiesTo = new Field();
        private Field _LinkToParentOrder = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID
        {
            get { return _SegmentTypeID; }
            private set { _SegmentTypeID = value; }
        }
        public Field SequenceNumber
        {
            get { return _SequenceNumber; }
            private set { _SequenceNumber = value; }
        }
        public Field UniqueForeignSpecimenID
        {
            get { return _UniqueForeignSpecimenID; }
            private set { _UniqueForeignSpecimenID = value; }
        }
        public Field ForeignApplicationID
        {
            get { return _ForeignApplicationID; }
            private set { _ForeignApplicationID = value; }
        }
        public Field InternalSpecimenID
        {
            get { return _InternalSpecimenID; }
            private set { _InternalSpecimenID = value; }
        }
        public Field InternalApplicationID
        {
            get { return _InternalApplicationID; }
            private set { _InternalApplicationID = value; }
        }
        public Field ObservationBatteryID
        {
            get { return _ObservationBatteryID; }
            private set { _ObservationBatteryID = value; }
        }
        public Field ObservationBatteryText
        {
            get { return _ObservationBatteryText; }
            private set { _ObservationBatteryText = value; }
        }
        public Field NameOfCodingSystem
        {
            get { return _NameOfCodingSystem; }
            private set { _NameOfCodingSystem = value; }
        }
        public Field Priority
        {
            get { return _Priority; }
            private set { _Priority = value; }
        }
        public Field RequestedDateTime
        {
            get { return _RequestedDateTime; }
            private set { _RequestedDateTime = value; }
        }
        public Field ObservationCollectionDateTime
        {
            get { return _ObservationCollectionDateTime; }
            private set { _ObservationCollectionDateTime = value; }
        }
        public Field ObservationCollectionEndDateTime
        {
            get { return _ObservationCollectionEndDateTime; }
            private set { _ObservationCollectionEndDateTime = value; }
        }
        public Field CollectionVolume
        {
            get { return _CollectionVolume; }
            private set { _CollectionVolume = value; }
        }
        public Field CollectorIdentifier
        {
            get { return _CollectorIdentifier; }
            private set { _CollectorIdentifier = value; }
        }
        public Field SpecimenActionCode
        {
            get { return _SpecimenActionCode; }
            private set { _SpecimenActionCode = value; }
        }
        public Field DangerCode
        {
            get { return _DangerCode; }
            private set { _DangerCode = value; }
        }
        public Field UnknownField
        {
            get { return _UnknownField; }
            private set { _UnknownField = value; }
        }
        public Field RelevantClinicalInformation
        {
            get { return _RelevantClinicalInformation; }
            private set { _RelevantClinicalInformation = value; }
        }
        public Field SpecimenReceivedDateTime
        {
            get { return _SpecimenReceivedDateTime; }
            private set { _SpecimenReceivedDateTime = value; }
        }
        public DateTime TypedSpecimenReceivedDateTime
        {
            get { return DateTime.Parse(_SpecimenReceivedDateTime.Value); }
        }
        public Field SpecimenSource
        {
            get { return _SpecimenSource; }
            private set { _SpecimenSource = value; }
        }
        public Field OrderingProvider
        {
            get { return _OrderingProvider; }
            private set { _OrderingProvider = value; }
        }
        public Field OrderingProviderPhoneNumber
        {
            get { return _OrderingProviderPhoneNumber; }
            private set { _OrderingProviderPhoneNumber = value; }
        }
        public Field AlternateUniqueForeignSpecimenID
        {
            get { return _AlternateUniqueForeignSpecimenID; }
            private set { _AlternateUniqueForeignSpecimenID = value; }
        }
        public Field RequestorField2
        {
            get { return _RequestorField2; }
            private set { _RequestorField2 = value; }
        }
        public Field ProducerField1
        {
            get { return _ProducerField1; }
            private set { _ProducerField1 = value; }
        }
        public Field ProducerField2
        {
            get { return _ProducerField2; }
            private set { _ProducerField2 = value; }
        }
        public Field DateTimeObservationReported
        {
            get { return _DateTimeObservationReported; }
            private set { _DateTimeObservationReported = value; }
        }
        public Field ProducerCharge
        {
            get { return _ProducerCharge; }
            private set { _ProducerCharge = value; }
        }
        public Field ProducerSectionID
        {
            get { return _ProducerSectionID; }
            private set { _ProducerSectionID = value; }
        }
        public Field OrderResultStatus
        {
            get { return _OrderResultStatus; }
            private set { _OrderResultStatus = value; }
        }
        public Field LinkToParentResult
        {
            get { return _LinkToParentResult; }
            private set { _LinkToParentResult = value; }
        }
        public Field QuantityTiming
        {
            get { return _QuantityTiming; }
            private set { _QuantityTiming = value; }
        }
        public Field ResultCopiesTo
        {
            get { return _ResultCopiesTo; }
            private set { _ResultCopiesTo = value; }
        }
        public Field LinkToParentOrder
        {
            get { return _LinkToParentOrder; }
            private set { _LinkToParentOrder = value; }
        }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get
            {
                return new List<Field>(new Field[] { SegmentTypeID, SequenceNumber, UniqueForeignSpecimenID, ForeignApplicationID, InternalSpecimenID, InternalApplicationID, ObservationBatteryID, ObservationBatteryText, NameOfCodingSystem, Priority, RequestedDateTime, ObservationCollectionDateTime, ObservationCollectionEndDateTime, CollectionVolume, CollectorIdentifier, SpecimenActionCode, DangerCode, UnknownField, RelevantClinicalInformation, SpecimenReceivedDateTime, SpecimenSource, OrderingProvider, OrderingProviderPhoneNumber, AlternateUniqueForeignSpecimenID, RequestorField2, ProducerField1, ProducerField2, DateTimeObservationReported, ProducerCharge, ProducerSectionID, OrderResultStatus, LinkToParentResult, QuantityTiming, ResultCopiesTo, LinkToParentOrder });
            }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public ObservationOrderSegment()
        {
            SegmentTypeID = new Field("Segment Type ID", "OBR.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            SequenceNumber = new Field("Sequence Number", "OBR.2", "To identify the number of OBR segments contained within the order file", RequiredFlags.AlwaysRequired, 4, string.Empty, SupportedCharacters.NumericOnly, 237, string.Empty);
            UniqueForeignSpecimenID = new Field("Unique Foreign Specimen ID", "OBR.3", "Customer specific accessioning or specimen identification number", RequiredFlags.AlwaysRequired, 30, string.Empty, SupportedCharacters.AlphaNumericSpecial, 216, string.Empty);
            ForeignApplicationID = new Field("Foreign Application ID", "OBR.3", "Constant value", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 217, string.Empty);
            InternalSpecimenID = new Field("Internal Specimen ID", "OBR.4", "Lab assigned specimen number", RequiredFlags.AlwaysRequired, 11, string.Empty, SupportedCharacters.AlphaNumericSpecial, 238, string.Empty);
            InternalApplicationID = new Field("Internal Application ID", "OBR.4", "Constant value", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 238, string.Empty);
            ObservationBatteryID = new Field("Observation Battery ID", "OBR.5", "Test number ordered", RequiredFlags.AlwaysRequired, 6, string.Empty, SupportedCharacters.AlphaNumericSpecial, 238, string.Empty);
            ObservationBatteryText = new Field("Observation Battery Text", "OBR.5", "Name of test ordered", RequiredFlags.AlwaysRequired, 30, string.Empty, SupportedCharacters.AlphaNumericSpecial, 238, string.Empty);
            NameOfCodingSystem = new Field("Name of Coding System", "OBR.5", "Constant value", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 238, string.Empty);
            Priority = new Field("Priority", "OBR.6", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 239, string.Empty);
            RequestedDateTime = new Field("Requested Date Time", "OBR.7", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 240, string.Empty);
            ObservationCollectionDateTime = new Field("Observation Collection Date Time", "OBR.8", "Record of the data and time the specimen was collected", RequiredFlags.Optional, 12, string.Empty, SupportedCharacters.NumericOnly, 241, string.Empty);
            ObservationCollectionEndDateTime = new Field("Observation Collection End Date Time", "OBR.9", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 242, string.Empty);
            CollectionVolume = new Field("Collection Volume", "OBR.10", "Amount of specimen collected", RequiredFlags.Conditional, 4, string.Empty, SupportedCharacters.NumericOnly, 243, string.Empty);
            CollectorIdentifier = new Field("Collector Identifier", "OBR.11", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 244, string.Empty);
            SpecimenActionCode = new Field("Specimen Action Code", "OBR.12", "To identify the type of result being returned", RequiredFlags.Conditional, 1, string.Empty, SupportedCharacters.AlphaOnly, 245, string.Empty);
            DangerCode = new Field("Danger Code", "OBR.13", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 246, string.Empty);
            UnknownField = new Field("Unknown Field", "OBR.14", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 247, string.Empty);
            RelevantClinicalInformation = new Field("Relevant Clinical Information", "OBR.14", "Informational Purposes", RequiredFlags.Conditional, 65, string.Empty, SupportedCharacters.AlphaNumericSpecial, 247, string.Empty);
            SpecimenReceivedDateTime = new Field("Specimen Received Date Time", "OBR.15", "Identify the order entry date and time", RequiredFlags.Optional, 14, string.Empty, SupportedCharacters.NumericOnly, 248, string.Empty);
            SpecimenSource = new Field("Specimen Source", "OBR.16", "Identifies the source of the specimen", RequiredFlags.Optional, 6, string.Empty, SupportedCharacters.AlphaNumericSpecial, 249, string.Empty);
            OrderingProvider = new Field("Ordering Provider", "OBR.17", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 226, string.Empty);
            OrderingProviderPhoneNumber = new Field("Ordering Provider Phone Number", "OBR.18", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 250, string.Empty);
            AlternateUniqueForeignSpecimenID = new Field("Alternate Unique Foreign Specimen ID", "OBR.19", "Alternate customer specific accessioning or specimen identification number", RequiredFlags.Optional, 11, string.Empty, SupportedCharacters.AlphaNumericSpecial, 251, string.Empty);
            RequestorField2 = new Field("Requestor Field 2", "OBR.20", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 252, string.Empty);
            ProducerField1 = new Field("Producer Field 1", "OBR.21", "Customer specific accessioning or specimen identification number", RequiredFlags.Optional, 30, string.Empty, SupportedCharacters.AlphaNumericSpecial, 253, string.Empty);
            ProducerField2 = new Field("Producer Field 2", "OBR.22", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 254, string.Empty);
            DateTimeObservationReported = new Field("Date Time Observation Reported", "OBR.23", "Identify the date and time the results were sent", RequiredFlags.AlwaysRequired, 14, string.Empty, SupportedCharacters.NumericOnly, 255, string.Empty);
            ProducerCharge = new Field("Producer Charge", "OBR.24", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 256, string.Empty);
            ProducerSectionID = new Field("Producer Section ID", "OBR.25", "Identification of the facility responsible for performing the testing", RequiredFlags.Optional, 5, string.Empty, SupportedCharacters.AlphaNumericSpecial, 257, string.Empty);
            OrderResultStatus = new Field("Order Result Status", "OBR.26", "Identification of status of results at the ordered item level", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 258, string.Empty);
            LinkToParentResult = new Field("Link to Parent Result", "OBR.27", "Identifies the test that reflexes creating this lab generated order", RequiredFlags.Conditional, 6, string.Empty, SupportedCharacters.AlphaNumericSpecial, 259, string.Empty);
            QuantityTiming = new Field("Quantity Timing", "OBR.28", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 221, string.Empty);
            ResultCopiesTo = new Field("Result Copies To", "OBR.29", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 260, string.Empty);
            LinkToParentOrder = new Field("Link to Parent Order", "OBR.30", "Identifies the ordered item containing the test that caused the reflex", RequiredFlags.Optional, 6, string.Empty, SupportedCharacters.AlphaNumericSpecial, 261, string.Empty);
        }
        public ObservationOrderSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            //The Order Message (ORM) only uses upto 19 field, why the count is 19 and not 29
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 19) { throw new ArgumentException("Not enough fields to construct a valid observation order segment (OBR).  Expected at least 19, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.SequenceNumber.Value = fields[1];
            if (fields[2].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'OBR.3'"); }
            List<string> components = new List<string>(fields[2].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned from field 'OBR.3'.  Expected 2, returned " + components.Count); }
            this.UniqueForeignSpecimenID.Value = components[0];
            this.ForeignApplicationID.Value = components[1];
            if (fields[3].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'OBR.4'"); }
            components = new List<string>(fields[3].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned from field 'OBR.4'.  Expected 2, returned " + components.Count); }
            this.InternalSpecimenID.Value = components[0];
            this.InternalApplicationID.Value = components[1];
            if (fields[4].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'OBR.5'"); }
            components = new List<string>(fields[4].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            if (components.Count != 3) { throw new ArgumentException("Unexpected number of components returned from field 'OBR.5'.  Expected 3, returned " + components.Count); }
            this.ObservationBatteryID.Value = components[0];
            this.ObservationBatteryText.Value = components[1];
            this.NameOfCodingSystem.Value = components[2];
            //Back to non-nested
            this.Priority.Value = fields[5];
            this.RequestedDateTime.Value = fields[6];
            this.ObservationCollectionDateTime.Value = fields[7];
            this.ObservationCollectionEndDateTime.Value = fields[8];
            this.CollectionVolume.Value = fields[9];
            this.CollectorIdentifier.Value = fields[10];
            this.SpecimenActionCode.Value = fields[11];
            this.DangerCode.Value = fields[12];
            //Nested
            if (fields[13].Contains(componentDelimiter) == true)//RelevantClinicalInformation is optional, so might not have a delimiter
            {
                components = new List<string>(fields[13].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
                if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned from field 'OBR.14'.  Expected 2, returned " + components.Count); }
                this.UnknownField.Value = components[0];
                this.RelevantClinicalInformation.Value = components[1];
            }
            else
            {
                this.UnknownField.Value = fields[13];
            }
            //Back to non-nested
            this.SpecimenReceivedDateTime.Value = fields[14];
            this.SpecimenSource.Value = fields[15];
            this.OrderingProvider.Value = fields[16];
            this.OrderingProviderPhoneNumber.Value = fields[17];
            this.AlternateUniqueForeignSpecimenID.Value = fields[18];
            //All these additional fields are used by the Result Message (ORU) only
            if (fields.Count > 19) { this.RequestorField2.Value = fields[19]; }
            if (fields.Count > 20) { this.ProducerField1.Value = fields[20];}
            if (fields.Count > 21) { this.ProducerField2.Value = fields[21];}
            if (fields.Count > 22) { this.DateTimeObservationReported.Value = fields[22];}
            if (fields.Count > 23) { this.ProducerCharge.Value = fields[23];}
            if (fields.Count > 24) { this.ProducerSectionID.Value = fields[24];}
            if (fields.Count > 25) { this.OrderResultStatus.Value = fields[25];}
            if (fields.Count > 26) { this.LinkToParentResult.Value = fields[26]; }
            if (fields.Count > 27) { this.ResultCopiesTo.Value = fields[27];}
            if (fields.Count > 28) { this.LinkToParentOrder.Value = fields[28]; }
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 29)
            {
                for (int x = 29; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "OBR." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Setting all field delimiters to the field delimiter, will change component ones below
            foreach (Field observationOrder in this.KnownFields)
            {
                observationOrder.Delimiter = fieldDelimiter;
            }
            //Components
            foreach (Field observationOrder in new Field[] { UniqueForeignSpecimenID, InternalSpecimenID, ObservationBatteryID, ObservationBatteryText, UnknownField })
            {
                observationOrder.Delimiter = componentDelimiter;
            }
            //Last field is always a newline
            LinkToParentOrder.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "OBR"; }
        }
        #endregion
    }
    public class ObservationResultSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _ValueType = new Field();
        private Field _ObservationIdentifier = new Field();
        private Field _ObservationText = new Field();
        private Field _NameOfCodingSystem = new Field();
        private Field _ObservationSubID = new Field();
        private Field _ObservationValue = new Field();
        private Field _UnitOfMeasureIdentifier = new Field();
        private Field _ReferenceRanges = new Field();
        private Field _AbnormalFlags = new Field();
        private Field _Probability = new Field();
        private Field _NatureOfAbnormal = new Field();
        private Field _ObservationResultStatus = new Field();
        private Field _DateOfLastChangeInReferenceRangeOrUnits = new Field();
        private Field _UserDefinedAccessChecks = new Field();
        private Field _DateTimeOfObservation = new Field();
        private Field _ProducerID = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID { get { return _SegmentTypeID; } private set { _SegmentTypeID = value; } }
        public Field SequenceNumber { get { return _SequenceNumber; } private set { _SequenceNumber = value; } }
        public Field ValueType { get { return _ValueType; } private set { _ValueType = value; } }
        public Field ObservationIdentifier { get { return _ObservationIdentifier; } private set { _ObservationIdentifier = value; } }
        public Field ObservationText { get { return _ObservationText; } private set { _ObservationText = value; } }
        public Field NameOfCodingSystem { get { return _NameOfCodingSystem; } private set { _NameOfCodingSystem = value; } }
        public Field ObservationSubID { get { return _ObservationSubID; } private set { _ObservationSubID = value; } }
        public Field ObservationValue { get { return _ObservationValue; } private set { _ObservationValue = value; } }
        public Field UnitOfMeasureIdentifier { get { return _UnitOfMeasureIdentifier; } private set { _UnitOfMeasureIdentifier = value; } }
        public Field ReferenceRanges { get { return _ReferenceRanges; } private set { _ReferenceRanges = value; } }
        public Field AbnormalFlags { get { return _AbnormalFlags; } private set { _AbnormalFlags = value; } }
        public Field Probability { get { return _Probability; } private set { _Probability = value; } }
        public Field NatureOfAbnormal { get { return _NatureOfAbnormal; } private set { _NatureOfAbnormal = value; } }
        public Field ObservationResultStatus { get { return _ObservationResultStatus; } private set { _ObservationResultStatus = value; } }
        public Field DateOfLastChangeInReferenceRangeOrUnits { get { return _DateOfLastChangeInReferenceRangeOrUnits; } private set { _DateOfLastChangeInReferenceRangeOrUnits = value; } }
        public DateTime TypedDateOfLastChangeInReferenceRangeOrUnits { get { return DateTime.Parse(DateOfLastChangeInReferenceRangeOrUnits.Value); } }
        public Field UserDefinedAccessChecks { get { return _UserDefinedAccessChecks; } private set { _UserDefinedAccessChecks = value; } }
        public Field DateTimeOfObservation { get { return _DateTimeOfObservation; } private set { _DateTimeOfObservation = value; } }
        public DateTime TypedDateTimeOfObservation { get { return DateTime.Parse(DateTimeOfObservation.Value); } }
        public Field ProducerID { get { return _ProducerID; } private set { _ProducerID = value; } }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get
            {
                return new List<Field>(new Field[] { SegmentTypeID, SequenceNumber, ValueType, ObservationIdentifier, ObservationText, NameOfCodingSystem, ObservationSubID, ObservationValue ,
            UnitOfMeasureIdentifier,ReferenceRanges,AbnormalFlags,Probability,NatureOfAbnormal,ObservationResultStatus,DateOfLastChangeInReferenceRangeOrUnits,UserDefinedAccessChecks,DateTimeOfObservation,ProducerID});
            }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public ObservationResultSegment()
        {
            SegmentTypeID = new Field("Segment Type ID", "OBX.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            SequenceNumber = new Field("Sequence Number", "OBX.2", "To identify the number of OBX segments contained within the order file", RequiredFlags.AlwaysRequired, 4, string.Empty, SupportedCharacters.NumericOnly, 569, string.Empty);
            ValueType = new Field("Value Type", "OBX.3", "To identify the types of data being sent", RequiredFlags.AlwaysRequired, 2, string.Empty, SupportedCharacters.AlphaOnly, 570, string.Empty);
            ObservationIdentifier = new Field("Observation Identifier", "OBX.4", "Test number", RequiredFlags.AlwaysRequired, 6, string.Empty, SupportedCharacters.AlphaNumericSpecial, 571, string.Empty);
            ObservationText = new Field("Observation Text", "OBX.4", "Test name", RequiredFlags.AlwaysRequired, 30, string.Empty, SupportedCharacters.AlphaNumericSpecial, 571, string.Empty);
            NameOfCodingSystem = new Field("Name of Coding System", "OBX.4", "Constant Value", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 571, string.Empty);
            ObservationSubID = new Field("Observation Sub-ID", "OBX.5", "To aid in the identification of results with the same observation ID underneath a given OBR", RequiredFlags.Optional, 2, string.Empty, SupportedCharacters.NumericOnly, 572, string.Empty);
            ObservationValue = new Field("Observation Value", "OBX.6", "Re[prted result", RequiredFlags.AlwaysRequired, 28, string.Empty, SupportedCharacters.AlphaNumericSpecial, 573, string.Empty);
            UnitOfMeasureIdentifier = new Field("Unit of Measure Identifier", "OBX.7", "To identify the units of measure", RequiredFlags.Conditional, 15, string.Empty, SupportedCharacters.AlphaNumericSpecial, 574, string.Empty);
            ReferenceRanges = new Field("Reference Ranges", "OBX.8", "To identify the reference ranges for the test performed", RequiredFlags.Conditional, 17, string.Empty, SupportedCharacters.AlphaNumericSpecial, 575, string.Empty);
            AbnormalFlags = new Field("Abnormal Flags", "OBX.9", "To identify if the result is an abnormal value for the test performed", RequiredFlags.Conditional, 2, string.Empty, SupportedCharacters.AlphaOnly, 576, string.Empty);
            Probability = new Field("Probability", "OBX.10", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 577, string.Empty);
            NatureOfAbnormal = new Field("Nature of Abnormal", "OBX.11", "Constant value", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 578, string.Empty);
            ObservationResultStatus = new Field("Observation Result Status", "OBX.12", "Status of result for the test", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 579, string.Empty);
            DateOfLastChangeInReferenceRangeOrUnits = new Field("Date of Last Change in Reference Range or Units", "OBX.13", "Informational purposes", RequiredFlags.Optional, 14, string.Empty, SupportedCharacters.NumericOnly, 580, string.Empty);
            UserDefinedAccessChecks = new Field("User-Defined Access Checks", "OBX.14", "Allows the producer to classify the result for the purpose of controlling user access in the case of sensitive testing", RequiredFlags.Optional, 15, string.Empty, SupportedCharacters.AlphaNumericSpecial, 581, string.Empty);
            DateTimeOfObservation = new Field("Date/Time of Observation", "OBX.15", "Identify the date and time the results were sent", RequiredFlags.Optional, 14, string.Empty, SupportedCharacters.NumericOnly, 582, string.Empty);
            ProducerID = new Field("Producer ID", "OBX.16", "Identification of the facility responsible for performing the testing", RequiredFlags.Conditional, 5, string.Empty, SupportedCharacters.AlphaNumericSpecial, 583, string.Empty);
        }
        public ObservationResultSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 16) { throw new ArgumentException("Not enough fields to construct a valid result observation segment (OBX).  Expected at least 16, returned " + fields.Count); }
            SegmentTypeID.Value = fields[0];
            SequenceNumber.Value = fields[1];
            ValueType.Value = fields[2];
            if (fields[3].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'OBX.4'"); }
            List<string> components = new List<string>(fields[3].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 3) { throw new ArgumentException("Unexpected number of components returned for field 'OBX.4'.  Expected 3, returned " + components.Count); }
            ObservationIdentifier.Value = components[0];
            ObservationText.Value = components[1];
            NameOfCodingSystem.Value = components[2];
            //Back to non-nested fields
            ObservationSubID.Value = fields[4];
            ObservationValue.Value = fields[5];
            UnitOfMeasureIdentifier.Value = fields[6];
            ReferenceRanges.Value = fields[7];
            AbnormalFlags.Value = fields[8];
            Probability.Value = fields[9];
            NatureOfAbnormal.Value = fields[10];
            ObservationResultStatus.Value = fields[11];
            DateOfLastChangeInReferenceRangeOrUnits.Value = fields[12];
            UserDefinedAccessChecks.Value = fields[13];
            DateTimeOfObservation.Value = fields[14];
            ProducerID.Value = fields[15];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 16)
            {
                for (int x = 16; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "OBX." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Non-nested field delimiters
            foreach (Field resultObservation in KnownFields)
            {
                resultObservation.Delimiter = fieldDelimiter;
            }
            //Nested field delimiters
            foreach (Field resultObservation in new Field[] { ObservationIdentifier, ObservationText })
            {
                resultObservation.Delimiter = componentDelimiter;
            }
            //last delimiter is always a newline
            ProducerID.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "OBX"; }
        }
        #endregion
    }
    public class NotesAndCommentsSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _CommentSource = new Field();
        private Field _CommentText = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID
        {
            get { return _SegmentTypeID; }
            private set { _SegmentTypeID = value; }
        }
        public Field SequenceNumber
        {
            get { return _SequenceNumber; }
            private set { _SequenceNumber = value; }
        }
        public Field CommentSource
        {
            get { return _CommentSource; }
            private set { _CommentSource = value; }
        }
        public Field CommentText
        {
            get { return _CommentText; }
            private set { _CommentText = value; }
        }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, SequenceNumber, CommentSource, CommentText }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public NotesAndCommentsSegment()
        {
            this.SegmentTypeID = new Field("Segment Type ID", "NTE.1", "To identify the segment", RequiredFlags.Optional, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.SequenceNumber = new Field("Sequence Number", "NTE.2", "To identify the numbet of NTE segments contained within the order file", RequiredFlags.Optional, 4, string.Empty, SupportedCharacters.NumericOnly, 96, string.Empty);
            this.CommentSource = new Field("Comment Source", "NTE.3", "To identify the source of the comment", RequiredFlags.Optional, 1, string.Empty, SupportedCharacters.AlphaOnly, 97, string.Empty);
            this.CommentText = new Field("Comment Text", "NTE.4", "Actual comment text", RequiredFlags.Optional, 78, string.Empty, SupportedCharacters.AlphaNumericSpecial, 98, string.Empty);
        }
        public NotesAndCommentsSegment(string message, string fieldDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 4) { throw new ArgumentException("Not enough fields to construct a valid notes and comments segment (NTE).  Expected at least 4, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.SequenceNumber.Value = fields[1];
            this.CommentSource.Value = fields[2];
            this.CommentText.Value = fields[3];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 4)
            {
                for (int x = 4; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "NTE." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiters
            foreach (Field noteAndComment in KnownFields)
            {
                noteAndComment.Delimiter = fieldDelimiter;
            }
            //Last one is always a newline
            this.CommentText.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "NTE"; }
        }
        #endregion
    }
    public class PlaceOfServiceFacilitySegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _FacilityMnemonic = new Field();
        private Field _FacilityName = new Field();
        private Field _FacilityAddress = new Field();
        private Field _FacilityOtherDesignation = new Field();
        private Field _FacilityCity = new Field();
        private Field _FacilityStateOrProvince = new Field();
        private Field _FacilityZipOrPostalCode = new Field();
        private Field _FacilityPhoneNumber = new Field();
        private Field _FacilityContact = new Field();
        private Field _FacilityDirectorTitle = new Field();
        private Field _FacilityDirectorLastName = new Field();
        private Field _FacilityDirectorFirstName = new Field();
        private Field _FacilityDirectorMiddleInitial = new Field();
        private Field _FacilityDirectorSuffix = new Field();
        private Field _FacilityDirectorPrefix = new Field();
        private Field _FacilityDirectorDegree = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID
        {
            get { return _SegmentTypeID; }
            private set { _SegmentTypeID = value; }
        }
        public Field SequenceNumber
        {
            get { return _SequenceNumber; }
            private set { _SequenceNumber = value; }
        }
        public Field FacilityMnemonic
        {
            get { return _FacilityMnemonic; }
            private set { _FacilityMnemonic = value; }
        }
        public Field FacilityName
        {
            get { return _FacilityName; }
            private set { _FacilityName = value; }
        }
        public Field FacilityAddress
        {
            get { return _FacilityAddress; }
            private set { _FacilityAddress = value; }
        }
        public Field FacilityOtherDesignation
        {
            get { return _FacilityOtherDesignation; }
            private set { _FacilityOtherDesignation = value; }
        }
        public Field FacilityCity
        {
            get { return _FacilityCity; }
            private set { _FacilityCity = value; }
        }
        public Field FacilityStateOrProvince
        {
            get { return _FacilityStateOrProvince; }
            private set { _FacilityStateOrProvince = value; }
        }
        public Field FacilityZipOrPostalCode
        {
            get { return _FacilityZipOrPostalCode; }
            private set { _FacilityZipOrPostalCode = value; }
        }
        public Field FacilityPhoneNumber
        {
            get { return _FacilityPhoneNumber; }
            private set { _FacilityPhoneNumber = value; }
        }
        public Field FacilityContact
        {
            get { return _FacilityContact; }
            private set { _FacilityContact = value; }
        }
        public Field FacilityDirectorTitle
        {
            get { return _FacilityDirectorTitle; }
            private set { _FacilityDirectorTitle = value; }
        }
        public Field FacilityDirectorLastName
        {
            get { return _FacilityDirectorLastName; }
            private set { _FacilityDirectorLastName = value; }
        }
        public Field FacilityDirectorFirstName
        {
            get { return _FacilityDirectorFirstName; }
            private set { _FacilityDirectorFirstName = value; }
        }
        public Field FacilityDirectorMiddleInitial
        {
            get { return _FacilityDirectorMiddleInitial; }
            private set { _FacilityDirectorMiddleInitial = value; }
        }
        public Field FacilityDirectorSuffix
        {
            get { return _FacilityDirectorSuffix; }
            private set { _FacilityDirectorSuffix = value; }
        }
        public Field FacilityDirectorPrefix
        {
            get { return _FacilityDirectorPrefix; }
            private set { _FacilityDirectorPrefix = value; }
        }
        public Field FacilityDirectorDegree
        {
            get { return _FacilityDirectorDegree; }
            private set { _FacilityDirectorDegree = value; }
        }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get
            {
                return new List<Field>(new Field[] { SegmentTypeID
                , SequenceNumber
                , FacilityMnemonic
                , FacilityName
                , FacilityAddress
                , FacilityOtherDesignation
                , FacilityCity
                , FacilityStateOrProvince
                , FacilityZipOrPostalCode
                , FacilityPhoneNumber
                , FacilityContact
                , FacilityDirectorTitle
                , FacilityDirectorLastName
                , FacilityDirectorFirstName
                , FacilityDirectorMiddleInitial
                , FacilityDirectorSuffix
                , FacilityDirectorPrefix
                , FacilityDirectorDegree});
            }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field placeOfService in this.AllFields)
                {
                    builder.Append(placeOfService.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field placeOfService in this.KnownFields)
                {
                    builder.Append(placeOfService.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public PlaceOfServiceFacilitySegment()
        {
            this.SegmentTypeID = new Field("Segment Type ID", "ZPS.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.SequenceNumber = new Field("Sequence Number", "ZPS.2", "To identify the number of ZPS segments contained within the order file", RequiredFlags.AlwaysRequired, 4, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            this.FacilityMnemonic = new Field("Facility Mnemonic", "ZPS.3", "Lab code of the lab that performed the testing", RequiredFlags.AlwaysRequired, 5, string.Empty, SupportedCharacters.AlphaNumericSpecial, 0, string.Empty);
            this.FacilityName = new Field("Facility Name", "ZPS.4", "Location that preformed the testing", RequiredFlags.AlwaysRequired, 30, string.Empty, SupportedCharacters.AlphaNumericSpecial, 0, string.Empty);
            this.FacilityAddress = new Field("Facility Address", "ZPS.5", "Informational Purposes", RequiredFlags.AlwaysRequired, 35, string.Empty, SupportedCharacters.AlphaNumericSpecial, 0, string.Empty);
            this.FacilityOtherDesignation = new Field("Facility Other Designation", "ZPS.5", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 0, string.Empty);
            this.FacilityCity = new Field("Facility City", "ZPS.5", "Informational Purposes", RequiredFlags.AlwaysRequired, 16, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.FacilityStateOrProvince = new Field("Facility State Or Province", "ZPS.5", "Informational Purposes", RequiredFlags.AlwaysRequired, 2, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.FacilityZipOrPostalCode = new Field("Facility Zip Or Postal Code", "ZPS.5", "Informational Purposes", RequiredFlags.AlwaysRequired, 9, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            this.FacilityPhoneNumber = new Field("Facility Phone Number", "ZPS.6", "Informational Purposes", RequiredFlags.Optional, 10, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            this.FacilityContact = new Field("Facility Contact", "ZPS.7", "", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 0, string.Empty);
            this.FacilityDirectorTitle = new Field("Facility Director Title", "ZPS.8", "Informational Purposes", RequiredFlags.Optional, 20, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.FacilityDirectorLastName = new Field("Facility Director Last Name", "ZPS.8", "Informational Purposes", RequiredFlags.AlwaysRequired, 20, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.FacilityDirectorFirstName = new Field("Facility Director First Name", "ZPS.8", "", RequiredFlags.AlwaysRequired, 20, "Informational Purposes", SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.FacilityDirectorMiddleInitial = new Field("Facility Director Middle Initial", "ZPS.8", "", RequiredFlags.Optional, 1, "Informational Purposes", SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.FacilityDirectorSuffix = new Field("Facility Director Suffix", "ZPS.8", "", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 0, string.Empty);
            this.FacilityDirectorPrefix = new Field("Facility Director Prefix", "ZPS.8", string.Empty, RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 0, string.Empty);
            this.FacilityDirectorDegree = new Field("Facility Director Degree", "ZPS.8", "Informational Purposes", RequiredFlags.Optional, 20, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
        }
        public PlaceOfServiceFacilitySegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 8) { throw new ArgumentException("Not enough fields to construct a valid place of service facility segment (ZPS).  Expected at least 8, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.SequenceNumber.Value = fields[1];
            this.FacilityMnemonic.Value = fields[2];
            this.FacilityName.Value = fields[3];
            if (fields[4].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ZPS.5'"); }
            List<string> components = new List<string>(fields[4].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 5) { throw new ArgumentException("Unexpected number of components returned from field 'ZPS.4'.  Expected 5, returned " + components.Count); }
            this.FacilityAddress.Value = components[0];
            this.FacilityOtherDesignation.Value = components[1];
            this.FacilityCity.Value = components[2];
            this.FacilityStateOrProvince.Value = components[3];
            this.FacilityZipOrPostalCode.Value = components[4];
            //Back to non-nested fields
            this.FacilityPhoneNumber.Value = fields[5];
            this.FacilityContact.Value = fields[6];
            //More nested fields
            if (fields[7].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ZPS.8'"); }
            components = new List<string>(fields[7].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 7) { throw new ArgumentException("Unexpected number of components returned from field 'ZPS.8'.  Expected 7, returned " + components.Count); }
            this.FacilityDirectorTitle.Value = components[0];
            this.FacilityDirectorLastName.Value = components[1];
            this.FacilityDirectorFirstName.Value = components[2];
            this.FacilityDirectorMiddleInitial.Value = components[3];
            this.FacilityDirectorSuffix.Value = components[4];
            this.FacilityDirectorPrefix.Value = components[5];
            this.FacilityDirectorDegree.Value = components[6];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 8)
            {
                for (int x = 8; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "ZPS." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiters
            foreach (Field facility in KnownFields)
            {
                facility.Delimiter = fieldDelimiter;
            }
            //Comonent (nested fields) Delimiters
            foreach (Field facility in new Field[] { FacilityAddress, FacilityOtherDesignation, FacilityCity, FacilityStateOrProvince, FacilityDirectorTitle, FacilityDirectorLastName, FacilityDirectorFirstName, FacilityDirectorMiddleInitial, FacilityDirectorSuffix, FacilityDirectorPrefix })
            {
                facility.Delimiter = componentDelimiter;
            }
            //Last field is always a newline
            FacilityDirectorDegree.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "ZPS"; }
        }
        #endregion
    }
    public class AddendumSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _ContinuationOfPreviousLogicalRecord = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID
        {
            get { return _SegmentTypeID; }
            private set { _SegmentTypeID = value; }
        }
        public Field ContinuationOfPreviousLogicalRecord
        {
            get { return _ContinuationOfPreviousLogicalRecord; }
            private set { _ContinuationOfPreviousLogicalRecord = value; }
        }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, ContinuationOfPreviousLogicalRecord }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public AddendumSegment()
        {
            this.SegmentTypeID = new Field("Segment Type ID", "ADD.1", "To identify the segment", RequiredFlags.Optional, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.ContinuationOfPreviousLogicalRecord = new Field("Continuation of Previous Logical Record", "ADD.2", "Continuation of Previous Logical Record", RequiredFlags.Optional, 215, string.Empty, SupportedCharacters.AlphaNumericSpecial, 0, string.Empty);
        }
        public AddendumSegment(string message, string fieldDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 2) { throw new ArgumentException("Not enough fields to construct a valid addendum segment (ADD).  Expected at least 2, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.ContinuationOfPreviousLogicalRecord.Value = fields[1];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 2)
            {
                for (int x = 2; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "ADD." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiter
            this.SegmentTypeID.Delimiter = fieldDelimiter;
            //Last one is always a newline
            this.ContinuationOfPreviousLogicalRecord.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "ADD"; }
        }
        #endregion
    }
    public class PatientVisitSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _PatientClass = new Field();
        private Field _AssignedPatientLocation = new Field();
        private Field _AdmissionType = new Field();
        private Field _PreAdmitNumber = new Field();
        private Field _PriorPatientLocation = new Field();
        private Field _PhysicianID = new Field();
        private Field _PhysicianLastName = new Field();
        private Field _PhysicianFirstInitial = new Field();
        private Field _PhysicianMiddleInitial = new Field();
        private Field _PhysicianSuffix = new Field();
        private Field _PhysicianPrefix = new Field();
        private Field _PhysicianDegree = new Field();
        private Field _SourceTable = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID { get { return _SegmentTypeID; } private set { _SegmentTypeID = value; } }
        public Field SequenceNumber { get { return _SequenceNumber; } private set { _SequenceNumber = value; } }
        public Field PatientClass { get { return _PatientClass; } private set { _PatientClass = value; } }
        public Field AssignedPatientLocation { get { return _AssignedPatientLocation; } private set { _AssignedPatientLocation = value; } }
        public Field AdmissionType { get { return _AdmissionType; } private set { _AdmissionType = value; } }
        public Field PreAdmitNumber { get { return _PreAdmitNumber; } private set { _PreAdmitNumber = value; } }
        public Field PriorPatientLocation { get { return _PriorPatientLocation; } private set { _PriorPatientLocation = value; } }
        public Field PhysicianID { get { return _PhysicianID; } private set { _PhysicianID = value; } }
        public Field PhysicianLastName { get { return _PhysicianLastName; } private set { _PhysicianLastName = value; } }
        public Field PhysicianFirstInitial { get { return _PhysicianFirstInitial; } private set { _PhysicianFirstInitial = value; } }
        public Field PhysicianMiddleInitial { get { return _PhysicianMiddleInitial; } private set { _PhysicianMiddleInitial = value; } }
        public Field PhysicianSuffix { get { return _PhysicianSuffix; } private set { _PhysicianSuffix = value; } }
        public Field PhysicianPrefix { get { return _PhysicianPrefix; } private set { _PhysicianPrefix = value; } }
        public Field PhysicianDegree { get { return _PhysicianDegree; } private set { _PhysicianDegree = value; } }
        public Field SourceTable { get { return _SourceTable; } private set { _SourceTable = value; } }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, SequenceNumber, PatientClass, AssignedPatientLocation, AdmissionType, PreAdmitNumber, PriorPatientLocation, PhysicianID, PhysicianLastName, PhysicianFirstInitial, PhysicianMiddleInitial, PhysicianSuffix, PhysicianPrefix, PhysicianDegree, SourceTable }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public PatientVisitSegment()
        {
            SegmentTypeID = new Field("	Segment Type ID", "PV1.1", "To identify the segment	", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaNumeric, 0, string.Empty);
            SequenceNumber = new Field("Sequence Number", "	PV1.2", "", RequiredFlags.AlwaysRequired, 4, string.Empty, SupportedCharacters.NumericOnly, 131, string.Empty);
            PatientClass = new Field("Patient Class", "	PV1.3", "Not currently saved, used, or returned	", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 132, string.Empty);
            AssignedPatientLocation = new Field("Assigned Patient Location", "	PV1.4", "Not currently saved, used, or returned	", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 133, string.Empty);
            AdmissionType = new Field("	Admission Type", "PV1.5", "", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 134, string.Empty);
            PreAdmitNumber = new Field("Pre-Admit Number", "PV1.6", "Not currently saved, used, or returned	", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 135, string.Empty);
            PriorPatientLocation = new Field("	Prior Patient Location", "PV1.7", "Not currently saved, used, or returned	", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 136, string.Empty);
            PhysicianID = new Field("Physician ID", "PV1.8", "", RequiredFlags.AlwaysRequired, 10, string.Empty, SupportedCharacters.AlphaNumericSpecial, 137, string.Empty);
            PhysicianLastName = new Field("	Physician Last Name	", "PV1.8", "", RequiredFlags.AlwaysRequired, 9, string.Empty, SupportedCharacters.AlphaOnly, 137, string.Empty);
            PhysicianFirstInitial = new Field("Physician First Initial", "PV1.8", "", RequiredFlags.Conditional, 1, string.Empty, SupportedCharacters.AlphaOnly, 137, string.Empty);
            PhysicianMiddleInitial = new Field("Physician Middle Initial", "PV1.8", "", RequiredFlags.Conditional, 1, string.Empty, SupportedCharacters.AlphaOnly, 137, string.Empty);
            PhysicianSuffix = new Field("Physician Suffix", "PV1.8	", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 137, string.Empty);
            PhysicianPrefix = new Field("Physician Prefix", "PV1.8	", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 137, string.Empty);
            PhysicianDegree = new Field("Physician Degree", "PV1.8	", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 137, string.Empty);
            SourceTable = new Field("Source Table", "PV1.8	", "Required for certain types of billing", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 137, string.Empty);
        }
        public PatientVisitSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 8) { throw new ArgumentException("Not enough fields to construct a valid patient visit segment (PV1).  Expected at least 8, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.SequenceNumber.Value = fields[1];
            this.PatientClass.Value = fields[2];
            this.AssignedPatientLocation.Value = fields[3];
            this.AdmissionType.Value = fields[4];
            this.PreAdmitNumber.Value = fields[5];
            this.PriorPatientLocation.Value = fields[6];
            if (fields[7].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'PV1.8'"); }
            List<string> components = new List<string>(fields[7].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 8) { throw new ArgumentException("Unexpected number of components returned for the field 'PV1.8'.  Expected 8, returned " + components.Count); }
            this.PhysicianID.Value = components[0];
            this.PhysicianLastName.Value = components[1];
            this.PhysicianFirstInitial.Value = components[2];
            this.PhysicianMiddleInitial.Value = components[3];
            this.PhysicianSuffix.Value = components[4];
            this.PhysicianPrefix.Value = components[5];
            this.PhysicianDegree.Value = components[6];
            this.SourceTable.Value = components[7];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 8)
            {
                for (int x = 8; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "PV1." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Non-nested field delimiters
            foreach (Field patientVisit in KnownFields)
            {
                patientVisit.Delimiter = fieldDelimiter;
            }
            //Nested field delimiters
            foreach (Field patientVisit in new Field[] { PhysicianID, PhysicianLastName, PhysicianFirstInitial, PhysicianMiddleInitial, PhysicianSuffix, PhysicianPrefix, PhysicianDegree })
            {
                patientVisit.Delimiter = componentDelimiter;
            }
            //last delimiter is always a newline
            this.SourceTable.Delimiter = System.Environment.NewLine;

        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "PV1"; }
        }
        #endregion
    }
    public class InsuranceSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _InsurancePlanID = new Field();
        private Field _InsuranceCompanyID = new Field();
        private Field _InsuranceCarrierCode = new Field();
        private Field _InsuranceCompanyName = new Field();
        private Field _InsuranceCompanyAddressLine1 = new Field();
        private Field _InsuranceCompanyAddressLine2 = new Field();
        private Field _InsuranceCompanyCity = new Field();
        private Field _InsuranceCompanyState = new Field();
        private Field _InsuranceCompanyZipCode = new Field();
        private Field _InsuranceCompanyContactName = new Field();
        private Field _InsuranceCompanyPhoneNumber = new Field();
        private Field _GroupNumberOfInsuredPatient = new Field();
        private Field _GroupName = new Field();
        private Field _InsuredsGroupEmployerID = new Field();
        private Field _InsuredsGroupEmployerName = new Field();
        private Field _PlanEffectiveDateTime = new Field();
        private Field _PlanExpirationDateTime = new Field();
        private Field _AuthorizationInformation = new Field();
        private Field _PlanType = new Field();
        private Field _NameOfInsured = new Field();
        private Field _GuarantorRelationshipToPatient = new Field();
        private Field _InsuredsDateOfBirth = new Field();
        private Field _InsuredsAddress = new Field();
        private Field _AssignmentOfBenefits = new Field();
        private Field _CoordinatorOfBenefits = new Field();
        private Field _PrimaryPayor = new Field();
        private Field _NoticeOfAdmissionCode = new Field();
        private Field _NoticeOfAdmissionDateTime = new Field();
        private Field _ReportOfEligibilityFlag = new Field();
        private Field _ReportOfEligibilityDateTime = new Field();
        private Field _ReleaseInformationCode = new Field();
        private Field _PreAdmitCertification = new Field();
        private Field _VerificationDateTime = new Field();
        private Field _VerificationBy = new Field();
        private Field _TypeOfAgreement = new Field();
        private Field _BillingStatus = new Field();
        private Field _LifetimeReserveDays = new Field();
        private Field _DelayBeforeLifetimeReserveDays = new Field();
        private Field _CompanyPlanCode = new Field();
        private Field _InsuranceNumber = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID { get { return _SegmentTypeID; } private set { _SegmentTypeID = value; } }
        public Field SequenceNumber { get { return _SequenceNumber; } private set { _SequenceNumber = value; } }
        public Field InsurancePlanID { get { return _InsurancePlanID; } private set { _InsurancePlanID = value; } }
        public Field InsuranceCompanyID { get { return _InsuranceCompanyID; } private set { _InsuranceCompanyID = value; } }
        public Field InsuranceCarrierCode { get { return _InsuranceCarrierCode; } private set { _InsuranceCarrierCode = value; } }
        public Field InsuranceCompanyName { get { return _InsuranceCompanyName; } private set { _InsuranceCompanyName = value; } }
        public Field InsuranceCompanyAddressLine1 { get { return _InsuranceCompanyAddressLine1; } private set { _InsuranceCompanyAddressLine1 = value; } }
        public Field InsuranceCompanyAddressLine2 { get { return _InsuranceCompanyAddressLine2; } private set { _InsuranceCompanyAddressLine2 = value; } }
        public Field InsuranceCompanyCity { get { return _InsuranceCompanyCity; } private set { _InsuranceCompanyCity = value; } }
        public Field InsuranceCompanyState { get { return _InsuranceCompanyState; } private set { _InsuranceCompanyState = value; } }
        public Field InsuranceCompanyZipCode { get { return _InsuranceCompanyZipCode; } private set { _InsuranceCompanyZipCode = value; } }
        public Field InsuranceCompanyContactName { get { return _InsuranceCompanyContactName; } private set { _InsuranceCompanyContactName = value; } }
        public Field InsuranceCompanyPhoneNumber { get { return _InsuranceCompanyPhoneNumber; } private set { _InsuranceCompanyPhoneNumber = value; } }
        public Field GroupNumberOfInsuredPatient { get { return _GroupNumberOfInsuredPatient; } private set { _GroupNumberOfInsuredPatient = value; } }
        public Field GroupName { get { return _GroupName; } private set { _GroupName = value; } }
        public Field InsuredsGroupEmployerID { get { return _InsuredsGroupEmployerID; } private set { _InsuredsGroupEmployerID = value; } }
        public Field InsuredsGroupEmployerName { get { return _InsuredsGroupEmployerName; } private set { _InsuredsGroupEmployerName = value; } }
        public Field PlanEffectiveDateTime { get { return _PlanEffectiveDateTime; } private set { _PlanEffectiveDateTime = value; } }
        public Field PlanExpirationDateTime { get { return _PlanExpirationDateTime; } private set { _PlanExpirationDateTime = value; } }
        public Field AuthorizationInformation { get { return _AuthorizationInformation; } private set { _AuthorizationInformation = value; } }
        public Field PlanType { get { return _PlanType; } private set { _PlanType = value; } }
        public Field NameOfInsured { get { return _NameOfInsured; } private set { _NameOfInsured = value; } }
        public Field GuarantorRelationshipToPatient { get { return _GuarantorRelationshipToPatient; } private set { _GuarantorRelationshipToPatient = value; } }
        public Field InsuredsDateOfBirth { get { return _InsuredsDateOfBirth; } private set { _InsuredsDateOfBirth = value; } }
        public Field InsuredsAddress { get { return _InsuredsAddress; } private set { _InsuredsAddress = value; } }
        public Field AssignmentOfBenefits { get { return _AssignmentOfBenefits; } private set { _AssignmentOfBenefits = value; } }
        public Field CoordinatorOfBenefits { get { return _CoordinatorOfBenefits; } private set { _CoordinatorOfBenefits = value; } }
        public Field PrimaryPayor { get { return _PrimaryPayor; } private set { _PrimaryPayor = value; } }
        public Field NoticeOfAdmissionCode { get { return _NoticeOfAdmissionCode; } private set { _NoticeOfAdmissionCode = value; } }
        public Field NoticeOfAdmissionDateTime { get { return _NoticeOfAdmissionDateTime; } private set { _NoticeOfAdmissionDateTime = value; } }
        public Field ReportOfEligibilityFlag { get { return _ReportOfEligibilityFlag; } private set { _ReportOfEligibilityFlag = value; } }
        public Field ReportOfEligibilityDateTime { get { return _ReportOfEligibilityDateTime; } private set { _ReportOfEligibilityDateTime = value; } }
        public Field ReleaseInformationCode { get { return _ReleaseInformationCode; } private set { _ReleaseInformationCode = value; } }
        public Field PreAdmitCertification { get { return _PreAdmitCertification; } private set { _PreAdmitCertification = value; } }
        public Field VerificationDateTime { get { return _VerificationDateTime; } private set { _VerificationDateTime = value; } }
        public Field VerificationBy { get { return _VerificationBy; } private set { _VerificationBy = value; } }
        public Field TypeOfAgreement { get { return _TypeOfAgreement; } private set { _TypeOfAgreement = value; } }
        public Field BillingStatus { get { return _BillingStatus; } private set { _BillingStatus = value; } }
        public Field LifetimeReserveDays { get { return _LifetimeReserveDays; } private set { _LifetimeReserveDays = value; } }
        public Field DelayBeforeLifetimeReserveDays { get { return _DelayBeforeLifetimeReserveDays; } private set { _DelayBeforeLifetimeReserveDays = value; } }
        public Field CompanyPlanCode { get { return _CompanyPlanCode; } private set { _CompanyPlanCode = value; } }
        public Field InsuranceNumber { get { return _InsuranceNumber; } private set { _InsuranceNumber = value; } }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, SequenceNumber, InsurancePlanID, InsuranceCompanyID, InsuranceCarrierCode, InsuranceCompanyName, InsuranceCompanyAddressLine1, InsuranceCompanyAddressLine2, InsuranceCompanyCity, InsuranceCompanyState, InsuranceCompanyZipCode, InsuranceCompanyContactName, InsuranceCompanyPhoneNumber, GroupNumberOfInsuredPatient, GroupName, InsuredsGroupEmployerID, InsuredsGroupEmployerName, PlanEffectiveDateTime, PlanExpirationDateTime, AuthorizationInformation, PlanType, NameOfInsured, GuarantorRelationshipToPatient, InsuredsDateOfBirth, InsuredsAddress, AssignmentOfBenefits, CoordinatorOfBenefits, PrimaryPayor, NoticeOfAdmissionCode, NoticeOfAdmissionDateTime, ReportOfEligibilityFlag, ReportOfEligibilityDateTime, ReleaseInformationCode, PreAdmitCertification, VerificationDateTime, VerificationBy, TypeOfAgreement, BillingStatus, LifetimeReserveDays, DelayBeforeLifetimeReserveDays, CompanyPlanCode, InsuranceNumber }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "IN1"; }
        }
        #endregion
        #region Constructors
        public InsuranceSegment()
        {
            SegmentTypeID = new Field("Segment Type ID", "IN1.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaNumeric, 0, string.Empty);
            SequenceNumber = new Field("Sequence Number", "IN1.2", "To identify the number of IN1 segments contained within the order file", RequiredFlags.AlwaysRequired, 4, string.Empty, SupportedCharacters.Unused, 426, string.Empty);
            InsurancePlanID = new Field("Insurance Plan ID ", "IN1.3", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 368, string.Empty);
            InsuranceCompanyID = new Field("Insurance Company Identification Number", "IN1.4", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 428, string.Empty);
            InsuranceCarrierCode = new Field("Insurance Carrier Code", "IN1.4", "routes billing to the appropriate Insurance/HMO company", RequiredFlags.AlwaysRequired, 5, string.Empty, SupportedCharacters.AlphaNumericSpecial, 428, string.Empty);
            InsuranceCompanyName = new Field("Insurance Company Name  ", "IN1.5", "To identify the appropriate Insurance/HMO company", RequiredFlags.Conditional, 30, string.Empty, SupportedCharacters.AlphaNumericSpecial, 429, string.Empty);
            InsuranceCompanyAddressLine1 = new Field("Insurance Company Address Line 1", "IN1.6", "To identify the appropriate Insurance/HMO company", RequiredFlags.Conditional, 35, string.Empty, SupportedCharacters.AlphaNumericSpecial, 430, string.Empty);
            InsuranceCompanyAddressLine2 = new Field("Insurance Company Address Line 2", "IN1.6", "To identify the appropriate Insurance/HMO company", RequiredFlags.Conditional, 35, string.Empty, SupportedCharacters.AlphaNumericSpecial, 430, string.Empty);
            InsuranceCompanyCity = new Field("Insurance Company City", "IN1.6", "To identify the appropriate Insurance/HMO company", RequiredFlags.Conditional, 16, string.Empty, SupportedCharacters.AlphaOnly, 430, string.Empty);
            InsuranceCompanyState = new Field("Insurance Company State ", "IN1.6", "To bill to the appropriate Insurance/HMO company", RequiredFlags.Conditional, 2, string.Empty, SupportedCharacters.AlphaOnly, 430, string.Empty);
            InsuranceCompanyZipCode = new Field("Insurance Company Zip Code", "IN1.6", "To bill to the appropriate Insurance/HMO company", RequiredFlags.Conditional, 9, string.Empty, SupportedCharacters.NumericOnly, 430, string.Empty);
            InsuranceCompanyContactName = new Field("Insurance Company Contact Name", "IN1.7", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 431, string.Empty);
            InsuranceCompanyPhoneNumber = new Field("Insurance Company Phone Number", "IN1.8", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 432, string.Empty);
            GroupNumberOfInsuredPatient = new Field("Group Number of Insured Patient ", "IN1.9", "Required for certain billing options", RequiredFlags.Conditional, 15, string.Empty, SupportedCharacters.AlphaNumericSpecial, 433, string.Empty);
            GroupName = new Field("Group Name", "IN1.10", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 434, string.Empty);
            InsuredsGroupEmployerID = new Field("Insureds Group Employer ID", "IN1.11", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 435, string.Empty);
            InsuredsGroupEmployerName = new Field("Insureds Group Employer Name", "IN1.12", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 436, string.Empty);
            PlanEffectiveDateTime = new Field("Plan Effective Date / Time", "IN1.13", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 437, string.Empty);
            PlanExpirationDateTime = new Field("Plan Expiration Date / Time", "IN1.14", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 438, string.Empty);
            AuthorizationInformation = new Field("Authorization Information", "IN1.15", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 439, string.Empty);
            PlanType = new Field("Plan Type", "IN1.16", "Designation of the insurance type this IN1 segment supports", RequiredFlags.AlwaysRequired, 2, string.Empty, SupportedCharacters.AlphaOnly, 440, string.Empty);
            NameOfInsured = new Field("Name of Insured", "IN1.17", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 441, string.Empty);
            GuarantorRelationshipToPatient = new Field("Guarantor Relationship to Patient ", "IN1.18", "To identify the relationship of the patient to the subscriber", RequiredFlags.Conditional, 1, string.Empty, SupportedCharacters.NumericOnly, 442, string.Empty);
            InsuredsDateOfBirth = new Field("Insureds Date of Birth", "IN1.19", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 443, string.Empty);
            InsuredsAddress = new Field("Insureds Address", "IN1.20", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 444, string.Empty);
            AssignmentOfBenefits = new Field("Assignment of Benefits", "IN1.21", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 445, string.Empty);
            CoordinatorOfBenefits = new Field("Coordinator of Benefits", "IN1.22", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 446, string.Empty);
            PrimaryPayor = new Field("Primary Payor", "IN1.23", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 447, string.Empty);
            NoticeOfAdmissionCode = new Field("Notice of Admission Code", "IN1.24", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 448, string.Empty);
            NoticeOfAdmissionDateTime = new Field("Notice of Admission Date / Time", "IN1.25", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 449, string.Empty);
            ReportOfEligibilityFlag = new Field("Report of Eligibility Flag", "IN1.26", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 450, string.Empty);
            ReportOfEligibilityDateTime = new Field("Report of Eligibility Date/Time", "IN1.27", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 451, string.Empty);
            ReleaseInformationCode = new Field("Release Information Code", "IN1.28", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 452, string.Empty);
            PreAdmitCertification = new Field("Pre-admit Certification", "IN1.29", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 453, string.Empty);
            VerificationDateTime = new Field("Verification Date/Time", "IN1.30", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 454, string.Empty);
            VerificationBy = new Field("Verification By", "IN1.31", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 455, string.Empty);
            TypeOfAgreement = new Field("Type of Agreement", "IN1.32", "Workers Compensation Flag", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 456, string.Empty);
            BillingStatus = new Field("Billing Status", "IN1.33", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 457, string.Empty);
            LifetimeReserveDays = new Field("Lifetime Reserve Days", "IN1.34", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 458, string.Empty);
            DelayBeforeLifetimeReserveDays = new Field("Delay before Lifetime Reserve Days", "IN1.35", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 459, string.Empty);
            CompanyPlanCode = new Field("Company Plan Code", "IN1.36", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 460, string.Empty);
            InsuranceNumber = new Field("Insurance Number", "IN1.37", "Required for certain billing options", RequiredFlags.Conditional, 15, string.Empty, SupportedCharacters.AlphaNumericSpecial, 461, string.Empty);
        }
        public InsuranceSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 37) { throw new ArgumentException("Not enough fields to construct a valid insurance segment (IN1).  Expected at least 37, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.SequenceNumber.Value = fields[1];
            this.InsurancePlanID.Value = fields[2];
            if (fields[3].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'IN1.4'"); }
            List<string> components = new List<string>(fields[3].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned for the field 'IN1.4'.  Expected 2, returned " + components.Count); }
            this.InsuranceCompanyID.Value = components[0];
            this.InsuranceCarrierCode.Value = components[1];
            //Back to a non-nested
            this.InsuranceCompanyName.Value = fields[4];
            //Nested again
            if (fields[5].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'IN1.6'"); }
            components = new List<string>(fields[5].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            if (components.Count != 5) { throw new ArgumentException("Unexpected number of components returned for the field 'IN1.6'.  Expected 2, returned " + components.Count); }
            this.InsuranceCompanyAddressLine1.Value = components[0];
            this.InsuranceCompanyAddressLine2.Value = components[1];
            this.InsuranceCompanyCity.Value = components[2];
            this.InsuranceCompanyState.Value = components[3];
            this.InsuranceCompanyZipCode.Value = components[4];
            //Back to non-nested
            this.InsuranceCompanyContactName.Value = fields[6];
            this.InsuranceCompanyPhoneNumber.Value = fields[7];
            this.GroupNumberOfInsuredPatient.Value = fields[8];
            this.GroupName.Value = fields[9];
            this.InsuredsGroupEmployerID.Value = fields[10];
            this.InsuredsGroupEmployerName.Value = fields[11];
            this.PlanEffectiveDateTime.Value = fields[12];
            this.PlanExpirationDateTime.Value = fields[13];
            this.AuthorizationInformation.Value = fields[14];
            this.PlanType.Value = fields[15];
            this.NameOfInsured.Value = fields[16];
            this.GuarantorRelationshipToPatient.Value = fields[17];
            this.InsuredsDateOfBirth.Value = fields[18];
            this.InsuredsAddress.Value = fields[19];
            this.AssignmentOfBenefits.Value = fields[20];
            this.CoordinatorOfBenefits.Value = fields[21];
            this.PrimaryPayor.Value = fields[22];
            this.NoticeOfAdmissionCode.Value = fields[23];
            this.NoticeOfAdmissionDateTime.Value = fields[24];
            this.ReportOfEligibilityFlag.Value = fields[25];
            this.ReportOfEligibilityDateTime.Value = fields[26];
            this.ReleaseInformationCode.Value = fields[27];
            this.PreAdmitCertification.Value = fields[28];
            this.VerificationDateTime.Value = fields[29];
            this.VerificationBy.Value = fields[30];
            this.TypeOfAgreement.Value = fields[31];
            this.BillingStatus.Value = fields[32];
            this.LifetimeReserveDays.Value = fields[33];
            this.DelayBeforeLifetimeReserveDays.Value = fields[34];
            this.CompanyPlanCode.Value = fields[35];
            this.InsuranceNumber.Value = fields[36];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 37)
            {
                for (int x = 37; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "IN1." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Non-nested field delimiters
            foreach (Field patientVisit in KnownFields)
            {
                patientVisit.Delimiter = fieldDelimiter;
            }
            //Nested field delimiters
            foreach (Field patientVisit in new Field[] { InsuranceCompanyID, InsuranceCompanyAddressLine1, InsuranceCompanyAddressLine2, InsuranceCompanyCity, InsuranceCompanyState })
            {
                patientVisit.Delimiter = componentDelimiter;
            }
            //last delimiter is always a newline
            this.InsuranceNumber.Delimiter = System.Environment.NewLine;
        }
        #endregion
    }
    public class GuarantorSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _GuarantorNumber = new Field();
        private Field _GuarantorLastName = new Field();
        private Field _GuarantorFirstName = new Field();
        private Field _GuarantorSpouseName = new Field();
        private Field _GuarantorAddress = new Field();
        private Field _GuarantorOtherDesignation = new Field();
        private Field _GuarantorCity = new Field();
        private Field _GuarantorState = new Field();
        private Field _GuarantorZipCode = new Field();
        private Field _GuarantorHomePhoneNumber = new Field();
        private Field _GuarantorWorkPhoneNumber = new Field();
        private Field _GuarantorDOB = new Field();
        private Field _GuarantorSex = new Field();
        private Field _GuarantorType = new Field();
        private Field _GuarantorRelationshipToPatient = new Field();
        private Field _GuarantorSocialSecurityNumber = new Field();
        private Field _GuarantorDateTimeBegin = new Field();
        private Field _GuarantorDateTimeEnd = new Field();
        private Field _GuarantorPriority = new Field();
        private Field _GuarantorsEmployerName = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID { get { return _SegmentTypeID; } private set { _SegmentTypeID = value; } }
        public Field SequenceNumber { get { return _SequenceNumber; } private set { _SequenceNumber = value; } }
        public Field GuarantorNumber { get { return _GuarantorNumber; } private set { _GuarantorNumber = value; } }
        public Field GuarantorLastName { get { return _GuarantorLastName; } private set { _GuarantorLastName = value; } }
        public Field GuarantorFirstName { get { return _GuarantorFirstName; } private set { _GuarantorFirstName = value; } }
        public Field GuarantorSpouseName { get { return _GuarantorSpouseName; } private set { _GuarantorSpouseName = value; } }
        public Field GuarantorAddress { get { return _GuarantorAddress; } private set { _GuarantorAddress = value; } }
        public Field GuarantorOtherDesignation { get { return _GuarantorOtherDesignation; } private set { _GuarantorOtherDesignation = value; } }
        public Field GuarantorCity { get { return _GuarantorCity; } private set { _GuarantorCity = value; } }
        public Field GuarantorState { get { return _GuarantorState; } private set { _GuarantorState = value; } }
        public Field GuarantorZipCode { get { return _GuarantorZipCode; } private set { _GuarantorZipCode = value; } }
        public Field GuarantorHomePhoneNumber { get { return _GuarantorHomePhoneNumber; } private set { _GuarantorHomePhoneNumber = value; } }
        public Field GuarantorWorkPhoneNumber { get { return _GuarantorWorkPhoneNumber; } private set { _GuarantorWorkPhoneNumber = value; } }
        public Field GuarantorDOB { get { return _GuarantorDOB; } private set { _GuarantorDOB = value; } }
        public Field GuarantorSex { get { return _GuarantorSex; } private set { _GuarantorSex = value; } }
        public Field GuarantorType { get { return _GuarantorType; } private set { _GuarantorType = value; } }
        public Field GuarantorRelationshipToPatient { get { return _GuarantorRelationshipToPatient; } private set { _GuarantorRelationshipToPatient = value; } }
        public Field GuarantorSocialSecurityNumber { get { return _GuarantorSocialSecurityNumber; } private set { _GuarantorSocialSecurityNumber = value; } }
        public Field GuarantorDateTimeBegin { get { return _GuarantorDateTimeBegin; } private set { _GuarantorDateTimeBegin = value; } }
        public Field GuarantorDateTimeEnd { get { return _GuarantorDateTimeEnd; } private set { _GuarantorDateTimeEnd = value; } }
        public Field GuarantorPriority { get { return _GuarantorPriority; } private set { _GuarantorPriority = value; } }
        public Field GuarantorsEmployerName { get { return _GuarantorsEmployerName; } private set { _GuarantorsEmployerName = value; } }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, SequenceNumber, GuarantorNumber, GuarantorLastName, GuarantorFirstName, GuarantorSpouseName, GuarantorAddress, GuarantorOtherDesignation, GuarantorCity, GuarantorState, GuarantorZipCode, GuarantorHomePhoneNumber, GuarantorWorkPhoneNumber, GuarantorDOB, GuarantorSex, GuarantorType, GuarantorRelationshipToPatient, GuarantorSocialSecurityNumber, GuarantorDateTimeBegin, GuarantorDateTimeEnd, GuarantorPriority, GuarantorsEmployerName, }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "GT1"; }
        }
        #endregion
        #region Constructors
        public GuarantorSegment()
        {
            SegmentTypeID = new Field("Segment Type ID", "GT1.1", "Identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaNumeric, 0, string.Empty);
            SequenceNumber = new Field("Sequence Number", "GT1.2", "To identify the number of GT1 segments contained within the order file", RequiredFlags.AlwaysRequired, 4, string.Empty, SupportedCharacters.NumericOnly, 405, string.Empty);
            GuarantorNumber = new Field("Guarantor Number", "GT1.3", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 406, string.Empty);
            GuarantorLastName = new Field("Guarantor Last Name", "GT1.4", "Responsible party identifier", RequiredFlags.AlwaysRequired, 12, string.Empty, SupportedCharacters.AlphaOnly, 407, string.Empty);
            GuarantorFirstName = new Field("Guarantor First Name", "GT1.4", "Responsible party identifier", RequiredFlags.AlwaysRequired, 9, string.Empty, SupportedCharacters.AlphaOnly, 407, string.Empty);
            GuarantorSpouseName = new Field("Guarantor Spouse Name", "GT1.5", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 408, string.Empty);
            GuarantorAddress = new Field("Guarantor Address", "GT1.6", "Responsible party demographics", RequiredFlags.AlwaysRequired, 35, string.Empty, SupportedCharacters.AlphaNumericSpecial, 409, string.Empty);
            GuarantorOtherDesignation = new Field("Guarantor Other Designation", "GT1.6", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 409, string.Empty);
            GuarantorCity = new Field("Guarantor City", "GT1.6", "Responsible party demographics", RequiredFlags.AlwaysRequired, 16, string.Empty, SupportedCharacters.AlphaOnly, 409, string.Empty);
            GuarantorState = new Field("Guarantor State", "GT1.6", "Responsible party demographics", RequiredFlags.AlwaysRequired, 2, string.Empty, SupportedCharacters.AlphaOnly, 409, string.Empty);
            GuarantorZipCode = new Field("Guarantor Zip Code", "GT1.6", "Responsible party demographics", RequiredFlags.AlwaysRequired, 9, string.Empty, SupportedCharacters.NumericOnly, 409, string.Empty);
            GuarantorHomePhoneNumber = new Field("Guarantor Home Phone Number", "GT1.7", "This phone number can be the Guarantor's or the Patient's phone number", RequiredFlags.AlwaysRequired, 10, string.Empty, SupportedCharacters.NumericOnly, 410, string.Empty);
            GuarantorWorkPhoneNumber = new Field("Guarantor Work Phone Number", "GT1.8", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 411, string.Empty);
            GuarantorDOB = new Field("Guarantor DOB", "GT1.9", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 412, string.Empty);
            GuarantorSex = new Field("Guarantor Sex", "GT1.10", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 413, string.Empty);
            GuarantorType = new Field("Guarantor Type", "GT1.11", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 414, string.Empty);
            GuarantorRelationshipToPatient = new Field("Guarantor Relationship to Patient", "GT1.12", "To identify the relationship of the patient to the subscriber", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.NumericOnly, 415, string.Empty);
            GuarantorSocialSecurityNumber = new Field("Guarantor Social Security Number", "GT1.13", "Used in outbound file only", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 416, string.Empty);
            GuarantorDateTimeBegin = new Field("Guarantor Date/Time Begin", "GT1.14", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 417, string.Empty);
            GuarantorDateTimeEnd = new Field("Guarantor Date/Time End", "GT1.15", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 418, string.Empty);
            GuarantorPriority = new Field("Guarantor Priority", "GT1.16", "Not currently saved, used, or returned", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 419, string.Empty);
            GuarantorsEmployerName = new Field("Guarantor's Employer Name", "GT1.17", "For demographic purposes and required for certain billing options", RequiredFlags.Conditional, 15, string.Empty, SupportedCharacters.AlphaNumericSpecial, 420, string.Empty);
        }
        public GuarantorSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 17) { throw new ArgumentException("Not enough fields to construct a valid guarantor segment (GT1).  Expected at least 17, returned " + fields.Count); }
            SegmentTypeID.Value = fields[0];
            SequenceNumber.Value = fields[1];
            GuarantorNumber.Value = fields[2];
            if (fields[3].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'GT1.4'"); }
            List<string> components = new List<string>(fields[3].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned for the field 'GT1.4'.  Expected 2, returned " + components.Count); }
            GuarantorLastName.Value = components[0];
            GuarantorFirstName.Value = components[1];
            //Back to non-nested
            GuarantorSpouseName.Value = fields[4];
            //More nested
            if (fields[5].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'GT1.6'"); }
            components = new List<string>(fields[5].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 5) { throw new ArgumentException("Unexpected number of components returned for the field 'GT1.6'.  Expected 5, returned " + components.Count); }
            GuarantorAddress.Value = components[0];
            GuarantorOtherDesignation.Value = components[1];
            GuarantorCity.Value = components[2];
            GuarantorState.Value = components[3];
            GuarantorZipCode.Value = components[4];
            //Back to non-nested
            GuarantorHomePhoneNumber.Value = fields[6];
            GuarantorWorkPhoneNumber.Value = fields[7];
            GuarantorDOB.Value = fields[8];
            GuarantorSex.Value = fields[9];
            GuarantorType.Value = fields[10];
            GuarantorRelationshipToPatient.Value = fields[11];
            GuarantorSocialSecurityNumber.Value = fields[12];
            GuarantorDateTimeBegin.Value = fields[13];
            GuarantorDateTimeEnd.Value = fields[14];
            GuarantorPriority.Value = fields[15];
            GuarantorsEmployerName.Value = fields[16];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 17)
            {
                for (int x = 17; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "GT1." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Non-nested field delimiters
            foreach (Field patientVisit in KnownFields)
            {
                patientVisit.Delimiter = fieldDelimiter;
            }
            //Nested field delimiters
            foreach (Field patientVisit in new Field[] { GuarantorLastName, GuarantorAddress, GuarantorOtherDesignation, GuarantorCity, GuarantorState })
            {
                patientVisit.Delimiter = componentDelimiter;
            }
            //last delimiter is always a newline
            this.GuarantorsEmployerName.Delimiter = System.Environment.NewLine;
        }
        #endregion
    }
    public class DiagnosisSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _SequenceNumber = new Field();
        private Field _DiagnosisCodingMethod = new Field();
        private Field _DiagnosisCode = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID
        {
            get { return _SegmentTypeID; }
            private set { _SegmentTypeID = value; }
        }
        public Field SequenceNumber
        {
            get { return _SequenceNumber; }
            private set { _SequenceNumber = value; }
        }
        public Field DiagnosisCodingMethod
        {
            get { return _DiagnosisCodingMethod; }
            private set { _DiagnosisCodingMethod = value; }
        }
        public Field DiagnosisCode
        {
            get { return _DiagnosisCode; }
            private set { _DiagnosisCode = value; }
        }
        public List<Field> AdditionalFields
        {
            get { return _AdditionalFields; }
            private set { _AdditionalFields = value; }
        }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, SequenceNumber, DiagnosisCodingMethod, DiagnosisCode }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public DiagnosisSegment()
        {
            this.SegmentTypeID = new Field("Segment Type ID", "DG1.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaNumeric, 0, string.Empty);
            this.SequenceNumber = new Field("Sequence Number", "DG1.2", "To identify the numbet of DG1 segments contained within the order file", RequiredFlags.AlwaysRequired, 4, string.Empty, SupportedCharacters.NumericOnly, 375, string.Empty);
            this.DiagnosisCodingMethod = new Field("Diagnosis Coding Method", "DG1.3", "To indicate a coding method", RequiredFlags.AlwaysRequired, 2, string.Empty, SupportedCharacters.AlphaNumeric, 376, string.Empty);
            this.DiagnosisCode = new Field("Diagnosis Code", "DG1.4", "Physician's indication of patient's diagnosis which is used by some payors to validate the need for ordering laboratory procedures", RequiredFlags.Optional, 8, string.Empty, SupportedCharacters.AlphaNumericSpecial, 377, string.Empty);
        }
        public DiagnosisSegment(string message, string fieldDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 4) { throw new ArgumentException("Not enough fields to construct a valid diagnosis segment (DG1).  Expected at least 4, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.SequenceNumber.Value = fields[1];
            this.DiagnosisCodingMethod.Value = fields[2];
            this.DiagnosisCode.Value = fields[3];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 4)
            {
                for (int x = 4; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "DG1." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiters
            foreach (Field diagnosis in KnownFields)
            {
                diagnosis.Delimiter = fieldDelimiter;
            }
            //Last one is always a newline
            this.DiagnosisCode.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "DG1"; }
        }
        #endregion
    }
    public class PatientGeneralClinicalInformationSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _Height = new Field();
        private Field _WeightValue = new Field();
        private Field _WeightUnitOfMeasure = new Field();
        private Field _CollectionValue = new Field();
        private Field _CollectionUnitOfMeasure = new Field();
        private Field _Fasting = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID { get { return _SegmentTypeID; } private set { _SegmentTypeID = value; } }
        public Field Height { get { return _Height; } private set { _Height = value; } }
        public Field WeightValue { get { return _WeightValue; } private set { _WeightValue = value; } }
        public Field WeightUnitOfMeasure { get { return _WeightUnitOfMeasure; } private set { _WeightUnitOfMeasure = value; } }
        public Field CollectionValue { get { return _CollectionValue; } private set { _CollectionValue = value; } }
        public Field CollectionUnitOfMeasure { get { return _CollectionUnitOfMeasure; } private set { _CollectionUnitOfMeasure = value; } }
        public Field Fasting { get { return _Fasting; } private set { _Fasting = value; } }
        public List<Field> AdditionalFields { get { return _AdditionalFields; } private set { _AdditionalFields = value; } }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, Height, WeightValue, WeightUnitOfMeasure, CollectionValue, CollectionUnitOfMeasure, Fasting }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public PatientGeneralClinicalInformationSegment()
        {
            this.SegmentTypeID = new Field("Segment Type ID", "ZCI.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.Height = new Field("Height", "ZCI.2", "Height", RequiredFlags.Unused, -1, string.Empty, SupportedCharacters.Unused, 0, string.Empty);
            this.WeightValue = new Field("Weight Value", "ZCI.3", "In specific testing procedures in calculation of result", RequiredFlags.Conditional, 3, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            this.WeightUnitOfMeasure = new Field("Weight Unit of Measure", "ZCI.3", "Indication of unit of measure", RequiredFlags.Conditional, 10, string.Empty, SupportedCharacters.AlphaNumericSpecial, 0, string.Empty);
            this.CollectionValue = new Field("Collection Value", "ZCI.4", "In specific testing procedures in calculation of result", RequiredFlags.Conditional, 4, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            this.CollectionUnitOfMeasure = new Field("Collection Unit of Measure", "ZCI.4", "Indication of unit of measure", RequiredFlags.Conditional, 10, string.Empty, SupportedCharacters.AlphaNumericSpecial, 0, string.Empty);
            this.Fasting = new Field("Fasting", "ZCI.5", "In specific testing procedures in calculation of result", RequiredFlags.Conditional, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
        }
        public PatientGeneralClinicalInformationSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 5) { throw new ArgumentException("Not enough fields to construct a valid patient general clinical information segment (ZCI).  Expected at least 5, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.Height.Value = fields[1];
            if (fields[2].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ZCI.3'"); }
            List<string> components = new List<string>(fields[2].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned for the field 'ZCI.3'.  Expected 2, returned " + components.Count); }
            WeightValue.Value = components[0];
            WeightUnitOfMeasure.Value = components[1];
            if (fields[3].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ZCI.4'"); }
            components = new List<string>(fields[3].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned for the field 'ZCI.4'.  Expected 2, returned " + components.Count); }
            CollectionValue.Value = components[0];
            CollectionUnitOfMeasure.Value = components[1];
            //Last one non-nested
            this.Fasting.Value = fields[4];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 5)
            {
                for (int x = 5; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "ZCI." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiters
            foreach (Field patientGeneralClinicalInformation in KnownFields)
            {
                patientGeneralClinicalInformation.Delimiter = fieldDelimiter;
            }
            foreach (Field patientGeneralClinicalInformation in new Field[] { WeightValue, CollectionValue })
            {
                patientGeneralClinicalInformation.Delimiter = componentDelimiter;
            }
            //Last one is always a newline
            this.Fasting.Delimiter = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "ZCI"; }
        }
        #endregion
    }
    public class BloodLeadSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _PatientRace = new Field();
        private Field _Hispanic = new Field();
        private Field _BloodLeadType = new Field();
        private Field _BloodLeadPurpose = new Field();
        private Field _BloodLeadCountyCodeNumber = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID { get { return _SegmentTypeID; } private set { _SegmentTypeID = value; } }
        public Field PatientRace { get { return _PatientRace; } private set { _PatientRace = value; } }
        public Field Hispanic { get { return _Hispanic; } private set { _Hispanic = value; } }
        public Field BloodLeadType { get { return _BloodLeadType; } private set { _BloodLeadType = value; } }
        public Field BloodLeadPurpose { get { return _BloodLeadPurpose; } private set { _BloodLeadPurpose = value; } }
        public Field BloodLeadCountyCodeNumber { get { return _BloodLeadCountyCodeNumber; } private set { _BloodLeadCountyCodeNumber = value; } }
        public List<Field> AdditionalFields { get { return _AdditionalFields; } private set { _AdditionalFields = value; } }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, PatientRace, Hispanic, BloodLeadType, BloodLeadPurpose, BloodLeadCountyCodeNumber }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructors
        public BloodLeadSegment()
        {
            this.SegmentTypeID = new Field("Segment Type ID", "ZBL.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.PatientRace = new Field("Patient Race", "ZBL.2", "Race is required for blood lead testing procedures", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            this.Hispanic = new Field("Hispanic", "ZBL.3", "This field is self-identification rather than scientific classification", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            this.BloodLeadType = new Field("Blood Lead Type", "ZBL.4", "Indication of how specimen was collected", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.BloodLeadPurpose = new Field("Blood Lead Purpose", "ZBL.5", "Indication of reason for testing", RequiredFlags.Optional, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            this.BloodLeadCountyCodeNumber = new Field("BloodLeadCountyCodeNumber", "ZBL.6", "For state reporting of blood lead results", RequiredFlags.Conditional, 10, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
        }
        public BloodLeadSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 6) { throw new ArgumentException("Not enough fields to construct a valid patient general clinical information segment (ZBL).  Expected at least 6, returned " + fields.Count); }
            this.SegmentTypeID.Value = fields[0];
            this.PatientRace.Value = fields[1];
            this.Hispanic.Value = fields[2];
            this.BloodLeadType.Value = fields[3];
            this.BloodLeadPurpose.Value = fields[4];
            this.BloodLeadCountyCodeNumber.Value = fields[5];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 6)
            {
                for (int x = 6; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "ZBL." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiters
            foreach (Field diagnosis in KnownFields)
            {
                diagnosis.Delimiter = fieldDelimiter;
            }
            //Last one is always a newline
            this.BloodLeadCountyCodeNumber.Value = System.Environment.NewLine;
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "ZBL"; }
        }
        #endregion
    }
    public class BethesdaCytologySegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _Cervical = new Field();
        private Field _Endocervical = new Field();
        private Field _LabiaVulva = new Field();
        private Field _Vaginal = new Field();
        private Field _Endometrial = new Field();
        private Field _SwabSpatula = new Field();
        private Field _BrushSpatula = new Field();
        private Field _SpatulaAlone = new Field();
        private Field _BrushAlone = new Field();
        private Field _BroomAlone = new Field();
        private Field _OtherCollectionTechnique = new Field();
        private Field _LMPMenoDate = new Field();
        private Field _None = new Field();
        private Field _Hyst = new Field();
        private Field _Coniza = new Field();
        private Field _ColpBX = new Field();
        private Field _LaserVap = new Field();
        private Field _Cyro = new Field();
        private Field _Radiation = new Field();
        private Field _DatesResults = new Field();
        private Field _Pregnant = new Field();
        private Field _Lactating = new Field();
        private Field _OralContraceptives = new Field();
        private Field _Menopausal = new Field();
        private Field _EstroRX = new Field();
        private Field _PMPBleeding = new Field();
        private Field _PostPart = new Field();
        private Field _IUD = new Field();
        private Field _AllOtherPat = new Field();
        private Field _Negative = new Field();
        private Field _Atypical = new Field();
        private Field _Dysplasia = new Field();
        private Field _CaInSitu = new Field();
        private Field _Invasive = new Field();
        private Field _OtherPreviousInformation = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID { get { return _SegmentTypeID; } private set { _SegmentTypeID = value; } }
        public Field Cervical { get { return _Cervical; } private set { _Cervical = value; } }
        public Field Endocervical { get { return _Endocervical; } private set { _Endocervical = value; } }
        public Field LabiaVulva { get { return _LabiaVulva; } private set { _LabiaVulva = value; } }
        public Field Vaginal { get { return _Vaginal; } private set { _Vaginal = value; } }
        public Field Endometrial { get { return _Endometrial; } private set { _Endometrial = value; } }
        public Field SwabSpatula { get { return _SwabSpatula; } private set { _SwabSpatula = value; } }
        public Field BrushSpatula { get { return _BrushSpatula; } private set { _BrushSpatula = value; } }
        public Field SpatulaAlone { get { return _SpatulaAlone; } private set { _SpatulaAlone = value; } }
        public Field BrushAlone { get { return _BrushAlone; } private set { _BrushAlone = value; } }
        public Field BroomAlone { get { return _BroomAlone; } private set { _BroomAlone = value; } }
        public Field OtherCollectionTechnique { get { return _OtherCollectionTechnique; } private set { _OtherCollectionTechnique = value; } }
        public Field LMPMenoDate { get { return _LMPMenoDate; } private set { _LMPMenoDate = value; } }
        public Field None { get { return _None; } private set { _None = value; } }
        public Field Hyst { get { return _Hyst; } private set { _Hyst = value; } }
        public Field Coniza { get { return _Coniza; } private set { _Coniza = value; } }
        public Field ColpBX { get { return _ColpBX; } private set { _ColpBX = value; } }
        public Field LaserVap { get { return _LaserVap; } private set { _LaserVap = value; } }
        public Field Cyro { get { return _Cyro; } private set { _Cyro = value; } }
        public Field Radiation { get { return _Radiation; } private set { _Radiation = value; } }
        public Field DatesResults { get { return _DatesResults; } private set { _DatesResults = value; } }
        public Field Pregnant { get { return _Pregnant; } private set { _Pregnant = value; } }
        public Field Lactating { get { return _Lactating; } private set { _Lactating = value; } }
        public Field OralContraceptives { get { return _OralContraceptives; } private set { _OralContraceptives = value; } }
        public Field Menopausal { get { return _Menopausal; } private set { _Menopausal = value; } }
        public Field EstroRX { get { return _EstroRX; } private set { _EstroRX = value; } }
        public Field PMPBleeding { get { return _PMPBleeding; } private set { _PMPBleeding = value; } }
        public Field PostPart { get { return _PostPart; } private set { _PostPart = value; } }
        public Field IUD { get { return _IUD; } private set { _IUD = value; } }
        public Field AllOtherPat { get { return _AllOtherPat; } private set { _AllOtherPat = value; } }
        public Field Negative { get { return _Negative; } private set { _Negative = value; } }
        public Field Atypical { get { return _Atypical; } private set { _Atypical = value; } }
        public Field Dysplasia { get { return _Dysplasia; } private set { _Dysplasia = value; } }
        public Field CaInSitu { get { return _CaInSitu; } private set { _CaInSitu = value; } }
        public Field Invasive { get { return _Invasive; } private set { _Invasive = value; } }
        public Field OtherPreviousInformation { get { return _OtherPreviousInformation; } private set { _OtherPreviousInformation = value; } }
        public List<Field> AdditionalFields { get { return _AdditionalFields; } private set { _AdditionalFields = value; } }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, Cervical, Endocervical, LabiaVulva, Vaginal, Endometrial, SwabSpatula, BrushSpatula, SpatulaAlone, BrushAlone, BroomAlone, OtherCollectionTechnique, LMPMenoDate, None, Hyst, Coniza, ColpBX, LaserVap, Cyro, Radiation, DatesResults, Pregnant, Lactating, OralContraceptives, Menopausal, EstroRX, PMPBleeding, PostPart, IUD, AllOtherPat, Negative, Atypical, Dysplasia, CaInSitu, Invasive, OtherPreviousInformation }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "ZCY"; }
        }
        #endregion
        #region Constructors
        public BethesdaCytologySegment()
        {
            SegmentTypeID = new Field("Segment Type ID", "ZCY.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Cervical = new Field("Cervical", "ZCY.2", "Gynecological source", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Endocervical = new Field("Endocervical", "ZCY.3", "Gynecological source", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            LabiaVulva = new Field("Labia-Vulva", "ZCY.4", "Gynecological source", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Vaginal = new Field("Vaginal", "ZCY.5", "Gynecological source", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Endometrial = new Field("Endometrial", "ZCY.6", "Gynecological source", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            SwabSpatula = new Field("Swab-Spatula", "ZCY.7", "Collection technique", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            BrushSpatula = new Field("Brush-Spatula", "ZCY.8", "Collection technique", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            SpatulaAlone = new Field("Spatula-Alone", "ZCY.9", "Collection technique", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            BrushAlone = new Field("Brush-Alone", "ZCY.10", "Collection technique", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            BroomAlone = new Field("Broom-Alone", "ZCY.11", "Collection technique", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            OtherCollectionTechnique = new Field("Other collection technique", "ZCY.12", "Collection technique", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            LMPMenoDate = new Field("LMP-Meno Date", "ZCY.13", "Date of patient's last menstrual period or menopausal date", RequiredFlags.AlwaysRequired, 8, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            None = new Field("None", "ZCY.14", "Previous treatment", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Hyst = new Field("Hyst", "ZCY.15", "Previous treatment", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Coniza = new Field("Coniza", "ZCY.16", "Previous treatment", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            ColpBX = new Field("Colp-BX", "ZCY.17", "Previous treatment", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            LaserVap = new Field("Laser-Vap", "ZCY.18", "Previous treatment", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Cyro = new Field("Cyro", "ZCY.19", "Previous treatment", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Radiation = new Field("Radiation", "ZCY.20", "Previous treatment", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            DatesResults = new Field("Dates-Results", "ZCY.21", "Previous cytology information", RequiredFlags.AlwaysRequired, 25, string.Empty, SupportedCharacters.AlphaNumericSpecial, 0, string.Empty);
            Pregnant = new Field("Pregnant", "ZCY.22", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Lactating = new Field("Lactating", "ZCY.23", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            OralContraceptives = new Field("Oral Contraceptives", "ZCY.24", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Menopausal = new Field("Menopausal", "ZCY.25", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            EstroRX = new Field("Estro-RX", "ZCY.26", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            PMPBleeding = new Field("PMP-Bleeding", "ZCY.27", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            PostPart = new Field("Post-Part", "ZCY.28", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            IUD = new Field("IUD", "ZCY.29", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            AllOtherPat = new Field("All-Other-Pat", "ZCY.30", "Other patient information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Negative = new Field("Negative", "ZCY.31", " Previous cytology information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Atypical = new Field("Atypical", "ZCY.32", " Previous cytology information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Dysplasia = new Field("Dysplasia", "ZCY.33", " Previous cytology information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            CaInSitu = new Field("Ca-In-Situ", "ZCY.34", " Previous cytology information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Invasive = new Field("Invasive", "ZCY.35", " Previous cytology information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            OtherPreviousInformation = new Field("Other Previous Information", "ZCY.36", " Previous cytology information", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
        }
        public BethesdaCytologySegment(string message, string fieldDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 36) { throw new ArgumentException("Not enough fields to construct a valid bethesda cytology segment (ZCY).  Expected at least 36, returned " + fields.Count); }
            SegmentTypeID.Value = fields[0];
            Cervical.Value = fields[1];
            Endocervical.Value = fields[2];
            LabiaVulva.Value = fields[3];
            Vaginal.Value = fields[4];
            Endometrial.Value = fields[5];
            SwabSpatula.Value = fields[6];
            BrushSpatula.Value = fields[7];
            SpatulaAlone.Value = fields[8];
            BrushAlone.Value = fields[9];
            BroomAlone.Value = fields[10];
            OtherCollectionTechnique.Value = fields[11];
            LMPMenoDate.Value = fields[12];
            None.Value = fields[13];
            Hyst.Value = fields[14];
            Coniza.Value = fields[15];
            ColpBX.Value = fields[16];
            LaserVap.Value = fields[17];
            Cyro.Value = fields[18];
            Radiation.Value = fields[19];
            DatesResults.Value = fields[20];
            Pregnant.Value = fields[21];
            Lactating.Value = fields[22];
            OralContraceptives.Value = fields[23];
            Menopausal.Value = fields[24];
            EstroRX.Value = fields[25];
            PMPBleeding.Value = fields[26];
            PostPart.Value = fields[27];
            IUD.Value = fields[28];
            AllOtherPat.Value = fields[29];
            Negative.Value = fields[30];
            Atypical.Value = fields[31];
            Dysplasia.Value = fields[32];
            CaInSitu.Value = fields[33];
            Invasive.Value = fields[34];
            OtherPreviousInformation.Value = fields[35];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 36)
            {
                for (int x = 36; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "ZCY." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiters
            foreach (Field bethesdaCytology in KnownFields)
            {
                bethesdaCytology.Delimiter = fieldDelimiter;
            }
            //Last one is always a newline
            this.OtherPreviousInformation.Delimiter = System.Environment.NewLine;
        }
        #endregion
    }
    public class SuperAlphaFetaProteinSegment
    {
        #region Private Instance Variables
        private Field _SegmentTypeID = new Field();
        private Field _InsulinDependent = new Field();
        private Field _GestationAgeWeeks = new Field();
        private Field _GestationalAgeDays = new Field();
        private Field _GestationalAgeDecimalForm = new Field();
        private Field _GestationalAgeDate = new Field();
        private Field _GestationalAgeCalculationMethodByLastMenstrualPeriod = new Field();
        private Field _LastMenstrualPeriodDate = new Field();
        private Field _GestationalAgeCalculationMethodByUltrasound = new Field();
        private Field _UltrasoundDate = new Field();
        private Field _GestationalAgeCalculationMethodByEstimatedDateOfDelivery = new Field();
        private Field _EstimatedDateOfDeliveryDate = new Field();
        private Field _TypeOfPregnancy = new Field();
        private Field _RoutineScreening = new Field();
        private Field _PreviousNeuralTubeDefects = new Field();
        private Field _AdvancedMaternalAge = new Field();
        private Field _HistoryOfDownSyndrome = new Field();
        private Field _HistoryOfCysticFibrosis = new Field();
        private Field _OtherIndications = new Field();
        private Field _HandWrittenAFPInfo = new Field();
        private Field _ReasonForRepeatElevated = new Field();
        private Field _EarlyGA = new Field();
        private Field _Hemolyzed = new Field();
        private List<Field> _AdditionalFields = new List<Field>();
        #endregion
        #region Public Instance Properties
        public Field SegmentTypeID { get { return _SegmentTypeID; } private set { _SegmentTypeID = value; } }
        public Field InsulinDependent { get { return _InsulinDependent; } private set { _InsulinDependent = value; } }
        public Field GestationAgeWeeks { get { return _GestationAgeWeeks; } private set { _GestationAgeWeeks = value; } }
        public Field GestationalAgeDays { get { return _GestationalAgeDays; } private set { _GestationalAgeDays = value; } }
        public Field GestationalAgeDecimalForm { get { return _GestationalAgeDecimalForm; } private set { _GestationalAgeDecimalForm = value; } }
        public Field GestationalAgeDate { get { return _GestationalAgeDate; } private set { _GestationalAgeDate = value; } }
        public Field GestationalAgeCalculationMethodByLastMenstrualPeriod { get { return _GestationalAgeCalculationMethodByLastMenstrualPeriod; } private set { _GestationalAgeCalculationMethodByLastMenstrualPeriod = value; } }
        public Field LastMenstrualPeriodDate { get { return _LastMenstrualPeriodDate; } private set { _LastMenstrualPeriodDate = value; } }
        public Field GestationalAgeCalculationMethodByUltrasound { get { return _GestationalAgeCalculationMethodByUltrasound; } private set { _GestationalAgeCalculationMethodByUltrasound = value; } }
        public Field UltrasoundDate { get { return _UltrasoundDate; } private set { _UltrasoundDate = value; } }
        public Field GestationalAgeCalculationMethodByEstimatedDateOfDelivery { get { return _GestationalAgeCalculationMethodByEstimatedDateOfDelivery; } private set { _GestationalAgeCalculationMethodByEstimatedDateOfDelivery = value; } }
        public Field EstimatedDateOfDeliveryDate { get { return _EstimatedDateOfDeliveryDate; } private set { _EstimatedDateOfDeliveryDate = value; } }
        public Field TypeOfPregnancy { get { return _TypeOfPregnancy; } private set { _TypeOfPregnancy = value; } }
        public Field RoutineScreening { get { return _RoutineScreening; } private set { _RoutineScreening = value; } }
        public Field PreviousNeuralTubeDefects { get { return _PreviousNeuralTubeDefects; } private set { _PreviousNeuralTubeDefects = value; } }
        public Field AdvancedMaternalAge { get { return _AdvancedMaternalAge; } private set { _AdvancedMaternalAge = value; } }
        public Field HistoryOfDownSyndrome { get { return _HistoryOfDownSyndrome; } private set { _HistoryOfDownSyndrome = value; } }
        public Field HistoryOfCysticFibrosis { get { return _HistoryOfCysticFibrosis; } private set { _HistoryOfCysticFibrosis = value; } }
        public Field OtherIndications { get { return _OtherIndications; } private set { _OtherIndications = value; } }
        public Field HandWrittenAFPInfo { get { return _HandWrittenAFPInfo; } private set { _HandWrittenAFPInfo = value; } }
        public Field ReasonForRepeatElevated { get { return _ReasonForRepeatElevated; } private set { _ReasonForRepeatElevated = value; } }
        public Field EarlyGA { get { return _EarlyGA; } private set { _EarlyGA = value; } }
        public Field Hemolyzed { get { return _Hemolyzed; } private set { _Hemolyzed = value; } }
        public List<Field> AdditionalFields { get { return _AdditionalFields; } private set { _AdditionalFields = value; } }
        public List<Field> AllFields
        {
            get
            {
                List<Field> allFields = new List<Field>();
                allFields.AddRange(AdditionalFields);
                allFields.AddRange(KnownFields);
                return allFields;
            }
        }
        public List<Field> KnownFields
        {
            get { return new List<Field>(new Field[] { SegmentTypeID, InsulinDependent, GestationAgeWeeks, GestationalAgeDays, GestationalAgeDecimalForm, GestationalAgeDate, GestationalAgeCalculationMethodByLastMenstrualPeriod, LastMenstrualPeriodDate, GestationalAgeCalculationMethodByUltrasound, UltrasoundDate, GestationalAgeCalculationMethodByEstimatedDateOfDelivery, EstimatedDateOfDeliveryDate, TypeOfPregnancy, RoutineScreening, PreviousNeuralTubeDefects, AdvancedMaternalAge, HistoryOfDownSyndrome, HistoryOfCysticFibrosis, OtherIndications, HandWrittenAFPInfo, ReasonForRepeatElevated, EarlyGA, Hemolyzed }); }
        }
        public string AllFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.AllFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        public string KnownFieldsHL7String
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (Field commonOrder in this.KnownFields)
                {
                    builder.Append(commonOrder.RightSideDelimitedValue);
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Public Class Properties
        public static string SegmentType
        {
            get { return "ZSA"; }
        }
        #endregion
        #region Constructors
        public SuperAlphaFetaProteinSegment()
        {
            SegmentTypeID = new Field("Segment Type ID", "ZSA.1", "To identify the segment", RequiredFlags.AlwaysRequired, 3, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            InsulinDependent = new Field("Insulin Dependent", "ZSA.2", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            GestationAgeWeeks = new Field("Gestation Age Weeks", "ZSA.3", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 2, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            GestationalAgeDays = new Field("Gestational Age Days", "ZSA.3", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            GestationalAgeDecimalForm = new Field("Gestational Age Decimal Form", "ZSA.3", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 4, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            GestationalAgeDate = new Field("Gestational Age Date", "ZSA.3", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 8, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            GestationalAgeCalculationMethodByLastMenstrualPeriod = new Field("Gestational Age Calculation Method by Last Menstrual Period ", "ZSA.4", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            LastMenstrualPeriodDate = new Field("Last Menstrual Period Date", "ZSA.4", "To provide relevant information pertaining to the order", RequiredFlags.Conditional, 8, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            GestationalAgeCalculationMethodByUltrasound = new Field("Gestational Age Calculation Method by Ultrasound", "ZSA.5", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            UltrasoundDate = new Field("Ultrasound Date", "ZSA.5", "To provide relevant information pertaining to the order", RequiredFlags.Conditional, 8, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            GestationalAgeCalculationMethodByEstimatedDateOfDelivery = new Field("Gestational Age Calculation Method by Estimated Date of Delivery", "ZSA.6", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            EstimatedDateOfDeliveryDate = new Field("Estimated Date of Delivery Date", "ZSA.6", "To provide relevant information pertaining to the order", RequiredFlags.Conditional, 8, string.Empty, SupportedCharacters.NumericOnly, 0, string.Empty);
            TypeOfPregnancy = new Field("Type of Pregnancy", "ZSA.7", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            RoutineScreening = new Field("Routine Screening ", "ZSA.8", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            PreviousNeuralTubeDefects = new Field("Previous Neural Tube Defects ", "ZSA.9", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            AdvancedMaternalAge = new Field("Advanced Maternal Age ", "ZSA.10", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            HistoryOfDownSyndrome = new Field("History of Down Syndrome", "ZSA.11", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            HistoryOfCysticFibrosis = new Field("History of Cystic Fibrosis", "ZSA.12", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            OtherIndications = new Field("Other Indications", "ZSA.13", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            HandWrittenAFPInfo = new Field("Hand Written AFP Info", "ZSA.14", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 20, string.Empty, SupportedCharacters.AlphaNumeric, 0, string.Empty);
            ReasonForRepeatElevated = new Field("Reason for Repeat: Elevated", "ZSA.15", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            EarlyGA = new Field("Early GA", "ZSA.16", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
            Hemolyzed = new Field("Hemolyzed", "ZSA.17", "To provide relevant information pertaining to the order", RequiredFlags.AlwaysRequired, 1, string.Empty, SupportedCharacters.AlphaOnly, 0, string.Empty);
        }
        public SuperAlphaFetaProteinSegment(string message, string fieldDelimiter, string componentDelimiter)
            : this()
        {
            List<string> fields = new List<string>(message.Split(new string[] { fieldDelimiter }, StringSplitOptions.None));
            if (fields.Count < 17) { throw new ArgumentException("Not enough fields to construct a valid super alpha feta protein segment (ZSA).  Expected at least 17, returned " + fields.Count); }
            SegmentTypeID.Value = fields[0];
            InsulinDependent.Value = fields[1];
            if (fields[2].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ZSA.3'"); }
            List<string> components = new List<string>(fields[2].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            //A field can have seperate components, essentially nested fields with a differnt delimiter
            if (components.Count != 4) { throw new ArgumentException("Unexpected number of components returned for the field 'ZSA.3'.  Expected 4, returned " + components.Count); }
            GestationAgeWeeks.Value = components[0];
            GestationalAgeDays.Value = components[1];
            GestationalAgeDecimalForm.Value = components[2];
            GestationalAgeDate.Value = components[3];
            if (fields[3].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ZSA.4'"); }
            components = new List<string>(fields[2].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned for the field 'ZSA.4'.  Expected 2, returned " + components.Count); }
            GestationalAgeCalculationMethodByLastMenstrualPeriod.Value = components[0];
            LastMenstrualPeriodDate.Value = components[1];
            if (fields[4].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ZSA.5'"); }
            components = new List<string>(fields[4].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned for the field 'ZSA.5'.  Expected 2, returned " + components.Count); }
            GestationalAgeCalculationMethodByUltrasound.Value = components[0];
            UltrasoundDate.Value = components[1];
            if (fields[5].Contains(componentDelimiter) == false) { throw new ArgumentException("Missing component delimiter in field 'ZSA.6'"); }
            components = new List<string>(fields[5].Split(new string[] { componentDelimiter }, StringSplitOptions.None));
            if (components.Count != 2) { throw new ArgumentException("Unexpected number of components returned for the field 'ZSA.6'.  Expected 2, returned " + components.Count); }
            GestationalAgeCalculationMethodByEstimatedDateOfDelivery.Value = components[0];
            EstimatedDateOfDeliveryDate.Value = components[1];
            //back to non-nested
            TypeOfPregnancy.Value = fields[6];
            RoutineScreening.Value = fields[7];
            PreviousNeuralTubeDefects.Value = fields[8];
            AdvancedMaternalAge.Value = fields[9];
            HistoryOfDownSyndrome.Value = fields[10];
            HistoryOfCysticFibrosis.Value = fields[11];
            OtherIndications.Value = fields[12];
            HandWrittenAFPInfo.Value = fields[13];
            ReasonForRepeatElevated.Value = fields[14];
            EarlyGA.Value = fields[15];
            Hemolyzed.Value = fields[16];
            //Any additional segments that we don't know about
            //Doesn't handle nested segments
            if (fields.Count > 5)
            {
                for (int x = 5; x < fields.Count - 1; x++)
                {
                    Field newField = new Field("Additional unknown field", "ZSA." + x, "Unknown", RequiredFlags.Unknown, -1, fieldDelimiter, SupportedCharacters.Unknown, -1, fields[x]);
                    AdditionalFields.Add(newField);
                }
            }
            //Field Delimiters
            foreach (Field superAlphaFetaProtein in KnownFields)
            {
                superAlphaFetaProtein.Delimiter = fieldDelimiter;
            }
            foreach (Field superAlphaFetaProtein in new Field[] { GestationAgeWeeks, GestationalAgeDays, GestationalAgeDecimalForm, GestationalAgeCalculationMethodByLastMenstrualPeriod, GestationalAgeCalculationMethodByUltrasound, GestationalAgeCalculationMethodByEstimatedDateOfDelivery })
            {
                superAlphaFetaProtein.Delimiter = componentDelimiter;
            }
            //Last one is always a newline
            this.Hemolyzed.Delimiter = System.Environment.NewLine;
        }
        #endregion
    }
    public class HL7Message
    {
        #region Private Instance Variables
        //The int is the segment's location within the message
        private SortedDictionary<int, Object> _HL7Segments = new SortedDictionary<int, Object>();
        #endregion
        #region Private Instance Methods
        public SortedDictionary<int, T> RetrieveSegmentSequence<T>()
        {
            SortedDictionary<int, T> dictionary = new SortedDictionary<int, T>();
                foreach (KeyValuePair<int, Object> pair in this.HL7Segments)
                {
                    if (pair.Value.GetType() == typeof(T)) { dictionary.Add(pair.Key, (T)pair.Value); }
                }
                return dictionary;
            }
        #endregion
        #region Public Instance Properties
        public SortedDictionary<int, MessageHeaderSegment> MessageHeaderSequence
        {
            get { return RetrieveSegmentSequence<MessageHeaderSegment>(); }
        }
        public SortedDictionary<int, PatientIdentificationSegment> PatientIdentificationSequence
        {
            get { return RetrieveSegmentSequence<PatientIdentificationSegment>(); }
        }
        public SortedDictionary<int, PatientVisitSegment> PatientVisitSequence
        {
            get { return RetrieveSegmentSequence<PatientVisitSegment>(); }
        }
        public SortedDictionary<int, InsuranceSegment> InsuranceSequence
        {
            get { return RetrieveSegmentSequence<InsuranceSegment>(); }
        }
        public SortedDictionary<int, GuarantorSegment> GuarantorSequence
        {
            get { return RetrieveSegmentSequence<GuarantorSegment>(); }
        }
        public SortedDictionary<int, DiagnosisSegment> DiagnosisSequence
        {
            get { return RetrieveSegmentSequence<DiagnosisSegment>(); }
        }
        public SortedDictionary<int, PatientGeneralClinicalInformationSegment> PatientGeneralClinicalInformationSequence
        {
            get { return RetrieveSegmentSequence<PatientGeneralClinicalInformationSegment>(); }
        }
        public SortedDictionary<int, CommonOrderSegment> CommonOrderSequence
        {
            get { return RetrieveSegmentSequence<CommonOrderSegment>(); }
        }
        public SortedDictionary<int, ObservationOrderSegment> ObservationOrderSequence
        {
            get { return RetrieveSegmentSequence<ObservationOrderSegment>(); }
        }
        public SortedDictionary<int, BloodLeadSegment> BloodLeadSequence
        {
            get { return RetrieveSegmentSequence<BloodLeadSegment>(); }
        }
        public SortedDictionary<int, BethesdaCytologySegment> BethesdaCytologySequence
        {
            get { return RetrieveSegmentSequence<BethesdaCytologySegment>(); }
        }
        public SortedDictionary<int, SuperAlphaFetaProteinSegment> SuperAlphaFetaProteinSequence
        {
            get { return RetrieveSegmentSequence<SuperAlphaFetaProteinSegment>(); }
        }
        public SortedDictionary<int, NotesAndCommentsSegment> NotesAndCommentsSequence
        {
            get { return RetrieveSegmentSequence<NotesAndCommentsSegment>(); }
        }
        public SortedDictionary<int, AddendumSegment> AddendumSequence
        {
            get { return RetrieveSegmentSequence<AddendumSegment>(); }
        }
        public SortedDictionary<int, ObservationResultSegment> ObservationResultSequence
        {
            get { return RetrieveSegmentSequence<ObservationResultSegment>(); }
        }
        public SortedDictionary<int, PlaceOfServiceFacilitySegment> PlaceOfServiceFacilitySequence
        {
            get { return RetrieveSegmentSequence<PlaceOfServiceFacilitySegment>(); }
        }
        public SortedDictionary<int, Object> HL7Segments
        {
            get { return _HL7Segments; }
            private set { _HL7Segments = value; }
        }
        public string KnownFieldsHL7Message
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (KeyValuePair<int, Object> pair in this.HL7Segments)
                {
                    builder.Append((string)pair.Value.GetType().GetProperty("KnownFieldsHL7String").GetValue(pair.Value, null));
                }
                return builder.ToString();
            }
        }
        public string AllFieldsHL7Message
        {
            get
            {
                StringBuilder builder = new StringBuilder();
                foreach (KeyValuePair<int, Object> pair in this.HL7Segments)
                {
                    builder.Append((string)pair.Value.GetType().GetProperty("AllFieldsHL7String").GetValue(pair.Value, null));
                }
                return builder.ToString();
            }
        }
        #endregion
        #region Constructor
        public HL7Message(string message)
        {
            if (message.Contains(MessageHeaderSegment.SegmentType) == false) { throw new ArgumentException("No Message Header Segment (MSH) in HL7 message."); }
            List<string> segments = new List<string>(message.Split(new string[] { System.Environment.NewLine }, StringSplitOptions.None));
            //So we have the field/component delimiters for the other segments
            MessageHeaderSegment messageHeader = new MessageHeaderSegment();
            for (int x = 1; x<=segments.Count;x++ )
            {
                string segment = segments[x - 1];
                if (segment.Length < 3) { throw new ArgumentException("Segment is too small (less than 3 characters): " + segment); }
                switch (segment.Substring(0, 3).ToUpper())
                {
                    case "MSH":
                        messageHeader = new MessageHeaderSegment(segment);
                        this.HL7Segments.Add(x, messageHeader);
                        break;
                    case "PID":
                        this.HL7Segments.Add(x, new PatientIdentificationSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "ORC":
                        this.HL7Segments.Add(x, new CommonOrderSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "OBR":
                        this.HL7Segments.Add(x, new ObservationOrderSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "OBX":
                        this.HL7Segments.Add(x, new ObservationResultSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "NTE":
                        this.HL7Segments.Add(x, new NotesAndCommentsSegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "ZPS":
                        this.HL7Segments.Add(x, new PlaceOfServiceFacilitySegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "ADD":
                        this.HL7Segments.Add(x, new AddendumSegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "PV1":
                        this.HL7Segments.Add(x, new PatientVisitSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "IN1":
                        this.HL7Segments.Add(x, new InsuranceSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "GT1":
                        this.HL7Segments.Add(x, new GuarantorSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "DG1":
                        this.HL7Segments.Add(x, new DiagnosisSegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "ZCI":
                        this.HL7Segments.Add(x, new PatientGeneralClinicalInformationSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "ZSA":
                        this.HL7Segments.Add(x, new SuperAlphaFetaProteinSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    case "ZCY":
                        this.HL7Segments.Add(x, new BethesdaCytologySegment(segment, messageHeader.FieldDelimiter.Value));
                        break;
                    case "ZBL":
                        this.HL7Segments.Add(x, new BloodLeadSegment(segment, messageHeader.FieldDelimiter.Value, messageHeader.ComponentDelimiter.Value));
                        break;
                    default:
                        throw new ArgumentException("Unknown HL7 string passed to HL7TypeFromString");
                }
            }
        }
        #endregion
    }
}